/* Floating point class for the HP-67 calculator. Actually, the types
 * are double. */


#ifndef DATATYPES_H
#define DATATYPES_H


#include "hp67.h"

/* A pure virtual class. All calculator data must be a sub-class of
 * this. All calculator data must be capable of printing its value to
 * a buffer, or of parsing a buffer for a value. */ 
/* Printing, it must print a certain number of digits in the indicated
 * display mode. Parsing must interpret the buffer and assign the
 * internal value to match, and return 'PARSE_OK' on success,
 * 'PARSE_BAD' if the parsing encounters problems. Parsing must ignore
 * any trailing whitespace or anything after a COMMENT character, but
 * it should make "*remainder" point to comments if they are present */
class Calcdatatype {
public:
/* Indicate in *rows and *cols how much screen space will be needed to
 * print a representation of this data */
  virtual void printvalue(int ndigits, enum disp_mode dmode, char *buffer, 
			  int bsize, int *rows, int *cols) const = 0;  

  virtual parse_codes parsevalue(char const *buffer, char **remainder) = 0;
};



/* Arithdatatypes are any data types on which standard arithmetic can
 * be performed. These could include floats, ints, and complex
 * values. */ 
/* All must be able to be assigned values from a double, an int, or a
 * pointer to Hashable, and must support all arithmetic (within
 * the class), transcendental, and trigonometric functions. */
/* They must also have defined increment, decrement, nonzero, and
   relational operators, and be able to supply a value for "pi" */
class Arithdatatype : public Calcdatatype {
public:
  virtual ~Arithdatatype(void) {};

// Ability to return an integer which approximates the value
  virtual inline int int_value(void) const = 0;

// Check whether a data value is valid
  virtual inline int value_ok(void) const = 0;

// Assignment from double or int
  virtual Arithdatatype &operator= (double from) = 0;
  virtual Arithdatatype &operator= (int from) = 0;
  virtual Arithdatatype &operator= (Arithdatatype const &from) = 0;

// Unary + and -
//  virtual Arithdatatype &operator+ (void) const = 0;
//  virtual Arithdatatype &operator- (void) const = 0;
  
// Arithmetic functions. In all cases, the argument forms the right-hand
// operand in regular typography. So, the value of the argument is
// added to, subtracted from, post-multiplied by, or divided into the
// value in the current cell.
  virtual Arithdatatype &plus(Arithdatatype const &a) = 0;
  virtual Arithdatatype &minus(Arithdatatype const &a) = 0;
  virtual Arithdatatype &times(Arithdatatype const &a) = 0;
  virtual Arithdatatype &div_by(Arithdatatype const &a) = 0;
  
// Trig and trans
  virtual Arithdatatype &sin(void) = 0;
  virtual Arithdatatype &cos(void) = 0;
  virtual Arithdatatype &tan(void) = 0;
  virtual Arithdatatype &arcsin(void) = 0;
  virtual Arithdatatype &arccos(void) = 0;
  virtual Arithdatatype &arctan(void) = 0;
  virtual Arithdatatype &sqrt(void) = 0;
  virtual Arithdatatype &square(void) = 0;
  virtual Arithdatatype &log_e(void) = 0;
  virtual Arithdatatype &exp_e(void) = 0;
  virtual Arithdatatype &log_10(void) = 0;
  virtual Arithdatatype &exp_10(void) = 0;
  virtual Arithdatatype &recip(void) = 0;
  virtual Arithdatatype &power(Arithdatatype const &exponent) = 0;
  virtual Arithdatatype &abs(void) = 0;
  virtual Arithdatatype &factorial(void) = 0;
  virtual void rec2polar(Arithdatatype *, Arithdatatype *) { }
  virtual void polar2rec(Arithdatatype *, Arithdatatype *) { }

// Increment / decrement
  virtual Arithdatatype &operator++ (void) = 0;
  virtual Arithdatatype &operator-- (void) = 0;

// Relational operators
  virtual bool operator! (void) const = 0;
  virtual bool operator== (Arithdatatype const &cmpr) const = 0;
  virtual bool operator!= (Arithdatatype const &cmpr) const = 0;
  virtual bool operator> (Arithdatatype const &cmpr) const = 0;
  virtual bool operator>= (Arithdatatype const &cmpr) const = 0;
  virtual bool operator< (Arithdatatype const &cmpr) const = 0;
  virtual bool operator<= (Arithdatatype const &cmpr) const = 0;
  virtual bool operator== (int const &cmpr) const = 0;
  virtual bool operator!= (int const &cmpr) const = 0;
  virtual bool operator> (int const &cmpr) const = 0;
  virtual bool operator< (int const &cmpr) const = 0;

};




#endif  /* DATATYPES_H */
