local atomType = {
   ordinary = 0,
   bigOperator = 1,
   binaryOperator = 2,
   relationalOperator = 3,
   openingSymbol = 4,
   closeSymbol = 5,
   punctuationSymbol = 6,
   inner = 7,
   overSymbol = 8,
   underSymbol = 9,
   accentSymbol = 10,
   radicalSymbol = 11,
   vcenter = 12,
}

local symbolDefaults = {}

local mkAtomType = {
   mathopen = atomType.openingSymbol,
   mathclose = atomType.closeSymbol,
   mathord = atomType.ordinary,
   mathfence = atomType.ordinary,
   mathalpha = atomType.ordinary,
   mathbin = atomType.binaryOperator,
   mathaccent = atomType.accentSymbol,
   mathaccentwide = atomType.accentSymbol,
   mathrel = atomType.relationalOperator,
   mathunder = atomType.underSymbol,
   mathover = atomType.overSymbol,
   mathop = atomType.ordinary,
}

local symbols = {}

-- TODO: Stuff 4th argument (description) into help string for symbol command
local function addSymbol (codepoint, command, typ, _)
   local str = luautf8.char(codepoint)
   symbols[command] = str
   symbolDefaults[str] = { atom = mkAtomType[typ] }
end

-- Generated from the unicode-math-table.tex in the unicode-math LaTeX package
-- <wspr.io/unicode-math>

addSymbol(0x00021, "mathexclam", "mathclose", "exclamation mark")
addSymbol(0x00023, "mathoctothorpe", "mathord", "number sign")
addSymbol(0x00024, "mathdollar", "mathord", "dollar sign")
addSymbol(0x00025, "mathpercent", "mathord", "percent sign")
addSymbol(0x00026, "mathampersand", "mathord", "ampersand")
addSymbol(0x00028, "lparen", "mathopen", "left parenthesis")
addSymbol(0x00029, "rparen", "mathclose", "right parenthesis")
addSymbol(0x0002B, "mathplus", "mathbin", "plus sign b:")
addSymbol(0x0002C, "mathcomma", "mathpunct", "comma")
addSymbol(0x0002E, "mathperiod", "mathord", "full stop, period")
addSymbol(0x0002F, "mathslash", "mathord", "solidus")
addSymbol(0x0003A, "mathcolon", "mathpunct", "colon")
addSymbol(0x0003B, "mathsemicolon", "mathpunct", "semicolon p:")
addSymbol(0x0003C, "less", "mathrel", "less-than sign r:")
addSymbol(0x0003D, "equal", "mathrel", "equals sign r:")
addSymbol(0x0003E, "greater", "mathrel", "greater-than sign r:")
addSymbol(0x0003F, "mathquestion", "mathord", "question mark")
addSymbol(0x00040, "mathatsign", "mathord", "commercial at")
addSymbol(0x0005B, "lbrack", "mathopen", "left square bracket")
addSymbol(0x0005C, "backslash", "mathord", "reverse solidus")
addSymbol(0x0005D, "rbrack", "mathclose", "right square bracket")
addSymbol(0x0007B, "lbrace", "mathopen", "left curly bracket")
addSymbol(0x0007C, "vert", "mathfence", "vertical bar")
addSymbol(0x0007D, "rbrace", "mathclose", "right curly bracket")
addSymbol(0x000A3, "mathsterling", "mathord", "pound sign")
addSymbol(0x000A5, "mathyen", "mathord", "yen sign")
addSymbol(0x000A7, "mathsection", "mathord", "section symbol")
addSymbol(0x000AC, "neg", "mathord", "/neg /lnot not sign")
addSymbol(0x000B1, "pm", "mathbin", "plus-or-minus sign")
addSymbol(0x000B6, "mathparagraph", "mathord", "paragraph symbol")
addSymbol(0x000B7, "cdotp", "mathbin", "/centerdot b: middle dot")
addSymbol(0x000D7, "times", "mathbin", "multiply sign")
addSymbol(0x000F0, "matheth", "mathalpha", "eth")
addSymbol(0x000F7, "div", "mathbin", "divide sign")
addSymbol(0x001B5, "Zbar", "mathord", "impedance (latin capital letter z with stroke)")
addSymbol(0x00300, "grave", "mathaccent", "grave accent")
addSymbol(0x00301, "acute", "mathaccent", "acute accent")
addSymbol(0x00302, "hat", "mathaccent", "circumflex accent")
addSymbol(0x00302, "widehat", "mathaccentwide", "circumflex accent")
addSymbol(0x00303, "tilde", "mathaccent", "tilde")
addSymbol(0x00303, "widetilde", "mathaccentwide", "tilde")
addSymbol(0x00304, "bar", "mathaccent", "macron")
addSymbol(0x00305, "overbar", "mathaccent", "overbar embellishment")
addSymbol(0x00305, "wideoverbar", "mathaccentwide", "stretchy overbar embellishment")
addSymbol(0x00306, "breve", "mathaccent", "breve")
addSymbol(0x00306, "widebreve", "mathaccentwide", "stretchy breve")
addSymbol(0x00307, "dot", "mathaccent", "dot above")
addSymbol(0x00308, "ddot", "mathaccent", "dieresis")
addSymbol(0x00309, "ovhook", "mathaccent", "combining hook above")
addSymbol(0x0030A, "ocirc", "mathaccent", "ring")
addSymbol(0x0030C, "check", "mathaccent", "caron")
addSymbol(0x0030C, "widecheck", "mathaccentwide", "stretchy caron")
addSymbol(0x00310, "candra", "mathaccent", "candrabindu (non-spacing)")
addSymbol(0x00312, "oturnedcomma", "mathaccent", "combining turned comma above")
addSymbol(0x00315, "ocommatopright", "mathaccent", "combining comma above right")
addSymbol(0x0031A, "droang", "mathaccent", "left angle above (non-spacing)")
addSymbol(0x00330, "wideutilde", "mathbotaccentwide", "under tilde accent (multiple characters and non-spacing)")
addSymbol(0x00332, "mathunderbar", "mathbotaccentwide", "combining low line")
addSymbol(0x00338, "notaccent", "mathaccentoverlay", "combining long solidus overlay")
addSymbol(0x0034D, "underleftrightarrow", "mathbotaccentwide", "underleftrightarrow accent")
addSymbol(0x00391, "mupAlpha", "mathalpha", "capital alpha, greek")
addSymbol(0x00392, "mupBeta", "mathalpha", "capital beta, greek")
addSymbol(0x00393, "mupGamma", "mathalpha", "capital gamma, greek")
addSymbol(0x00394, "mupDelta", "mathalpha", "capital delta, greek")
addSymbol(0x00395, "mupEpsilon", "mathalpha", "capital epsilon, greek")
addSymbol(0x00396, "mupZeta", "mathalpha", "capital zeta, greek")
addSymbol(0x00397, "mupEta", "mathalpha", "capital eta, greek")
addSymbol(0x00398, "mupTheta", "mathalpha", "capital theta, greek")
addSymbol(0x00399, "mupIota", "mathalpha", "capital iota, greek")
addSymbol(0x0039A, "mupKappa", "mathalpha", "capital kappa, greek")
addSymbol(0x0039B, "mupLambda", "mathalpha", "capital lambda, greek")
addSymbol(0x0039C, "mupMu", "mathalpha", "capital mu, greek")
addSymbol(0x0039D, "mupNu", "mathalpha", "capital nu, greek")
addSymbol(0x0039E, "mupXi", "mathalpha", "capital xi, greek")
addSymbol(0x0039F, "mupOmicron", "mathalpha", "capital omicron, greek")
addSymbol(0x003A0, "mupPi", "mathalpha", "capital pi, greek")
addSymbol(0x003A1, "mupRho", "mathalpha", "capital rho, greek")
addSymbol(0x003A3, "mupSigma", "mathalpha", "capital sigma, greek")
addSymbol(0x003A4, "mupTau", "mathalpha", "capital tau, greek")
addSymbol(0x003A5, "mupUpsilon", "mathalpha", "capital upsilon, greek")
addSymbol(0x003A6, "mupPhi", "mathalpha", "capital phi, greek")
addSymbol(0x003A7, "mupChi", "mathalpha", "capital chi, greek")
addSymbol(0x003A8, "mupPsi", "mathalpha", "capital psi, greek")
addSymbol(0x003A9, "mupOmega", "mathalpha", "capital omega, greek")
addSymbol(0x003B1, "mupalpha", "mathalpha", "small alpha, greek")
addSymbol(0x003B2, "mupbeta", "mathalpha", "small beta, greek")
addSymbol(0x003B3, "mupgamma", "mathalpha", "small gamma, greek")
addSymbol(0x003B4, "mupdelta", "mathalpha", "small delta, greek")
addSymbol(0x003B5, "mupvarepsilon", "mathalpha", "rounded small varepsilon, greek")
addSymbol(0x003B6, "mupzeta", "mathalpha", "small zeta, greek")
addSymbol(0x003B7, "mupeta", "mathalpha", "small eta, greek")
addSymbol(0x003B8, "muptheta", "mathalpha", "straight theta, small theta, greek")
addSymbol(0x003B9, "mupiota", "mathalpha", "small iota, greek")
addSymbol(0x003BA, "mupkappa", "mathalpha", "small kappa, greek")
addSymbol(0x003BB, "muplambda", "mathalpha", "small lambda, greek")
addSymbol(0x003BC, "mupmu", "mathalpha", "small mu, greek")
addSymbol(0x003BD, "mupnu", "mathalpha", "small nu, greek")
addSymbol(0x003BE, "mupxi", "mathalpha", "small xi, greek")
addSymbol(0x003BF, "mupomicron", "mathalpha", "small omicron, greek")
addSymbol(0x003C0, "muppi", "mathalpha", "small pi, greek")
addSymbol(0x003C1, "muprho", "mathalpha", "small rho, greek")
addSymbol(0x003C2, "mupvarsigma", "mathalpha", "terminal sigma, greek")
addSymbol(0x003C3, "mupsigma", "mathalpha", "small sigma, greek")
addSymbol(0x003C4, "muptau", "mathalpha", "small tau, greek")
addSymbol(0x003C5, "mupupsilon", "mathalpha", "small upsilon, greek")
addSymbol(0x003C6, "mupvarphi", "mathalpha", "curly or open small phi, greek")
addSymbol(0x003C7, "mupchi", "mathalpha", "small chi, greek")
addSymbol(0x003C8, "muppsi", "mathalpha", "small psi, greek")
addSymbol(0x003C9, "mupomega", "mathalpha", "small omega, greek")
addSymbol(0x003D1, "mupvartheta", "mathalpha", "/vartheta - curly or open theta")
addSymbol(0x003D5, "mupphi", "mathalpha", "/straightphi - small phi, greek")
addSymbol(0x003D6, "mupvarpi", "mathalpha", "rounded small pi (pomega), greek")
addSymbol(0x003DC, "upDigamma", "mathalpha", "capital digamma")
addSymbol(0x003DD, "updigamma", "mathalpha", "old greek small letter digamma")
addSymbol(0x003F0, "mupvarkappa", "mathalpha", "rounded small kappa, greek")
addSymbol(0x003F1, "mupvarrho", "mathalpha", "rounded small rho, greek")
addSymbol(0x003F4, "mupvarTheta", "mathalpha", "greek capital theta symbol")
addSymbol(0x003F5, "mupepsilon", "mathalpha", "greek lunate varepsilon symbol")
addSymbol(0x003F6, "upbackepsilon", "mathord", "greek reversed lunate epsilon symbol")
addSymbol(0x02010, "mathhyphen", "mathalpha", "hyphen")
addSymbol(0x02015, "horizbar", "mathord", "horizontal bar")
addSymbol(0x02016, "Vert", "mathfence", "double vertical bar")
addSymbol(0x02017, "twolowline", "mathord", "double low line (spacing)")
addSymbol(0x02020, "dagger", "mathbin", "dagger relation")
addSymbol(0x02021, "ddagger", "mathbin", "double dagger relation")
addSymbol(0x02022, "smblkcircle", "mathbin", "/bullet b: round bullet, filled")
addSymbol(0x02025, "enleadertwodots", "mathord", "double baseline dot (en leader)")
addSymbol(0x02026, "unicodeellipsis", "mathord", "ellipsis (horizontal)")
addSymbol(0x02032, "prime", "mathord", "prime or minute, not superscripted")
addSymbol(0x02033, "dprime", "mathord", "double prime or second, not superscripted")
addSymbol(0x02034, "trprime", "mathord", "triple prime (not superscripted)")
addSymbol(0x02035, "backprime", "mathord", "reverse prime, not superscripted")
addSymbol(0x02036, "backdprime", "mathord", "double reverse prime, not superscripted")
addSymbol(0x02037, "backtrprime", "mathord", "triple reverse prime, not superscripted")
addSymbol(0x02038, "caretinsert", "mathord", "caret (insertion mark)")
addSymbol(0x0203C, "Exclam", "mathord", "double exclamation mark")
addSymbol(0x02040, "tieconcat", "mathbin", "character tie, z notation sequence concatenation")
addSymbol(0x02043, "hyphenbullet", "mathord", "rectangle, filled (hyphen bullet)")
addSymbol(0x02044, "fracslash", "mathbin", "fraction slash")
addSymbol(0x02047, "Question", "mathord", "double question mark")
addSymbol(0x02050, "closure", "mathrel", "close up")
addSymbol(0x02057, "qprime", "mathord", "quadruple prime, not superscripted")
addSymbol(0x020AC, "euro", "mathord", "euro sign")
addSymbol(0x020D0, "leftharpoonaccent", "mathaccent", "combining left harpoon above")
addSymbol(0x020D0, "overleftharpoon", "mathaccentwide", "combining left harpoon above")
addSymbol(0x020D1, "rightharpoonaccent", "mathaccent", "combining right harpoon above")
addSymbol(0x020D1, "overrightharpoon", "mathaccentwide", "combining right harpoon above")
addSymbol(0x020D2, "vertoverlay", "mathaccent", "combining long vertical line overlay")
addSymbol(0x020D6, "overleftarrow", "mathaccentwide", "combining left arrow above")
addSymbol(0x020D7, "overrightarrow", "mathaccentwide", "combining left arrow above")
addSymbol(0x020D7, "vec", "mathaccent", "combining right arrow above")
addSymbol(0x020DB, "dddot", "mathaccent", "combining three dots above")
addSymbol(0x020DC, "ddddot", "mathaccent", "combining four dots above")
addSymbol(0x020DD, "enclosecircle", "mathord", "combining enclosing circle")
addSymbol(0x020DE, "enclosesquare", "mathord", "combining enclosing square")
addSymbol(0x020DF, "enclosediamond", "mathord", "combining enclosing diamond")
addSymbol(0x020E1, "overleftrightarrow", "mathaccentwide", "combining left right arrow above")
addSymbol(0x020E4, "enclosetriangle", "mathord", "combining enclosing upward pointing triangle")
addSymbol(0x020E7, "annuity", "mathaccent", "combining annuity symbol")
addSymbol(0x020E8, "threeunderdot", "mathbotaccent", "combining triple underdot")
addSymbol(0x020E9, "widebridgeabove", "mathaccent", "combining wide bridge above")
addSymbol(0x020EC, "underrightharpoondown", "mathbotaccentwide", "combining rightwards harpoon with barb downwards")
addSymbol(0x020ED, "underleftharpoondown", "mathbotaccentwide", "combining leftwards harpoon with barb downwards")
addSymbol(0x020EE, "underleftarrow", "mathbotaccentwide", "combining left arrow below")
addSymbol(0x020EF, "underrightarrow", "mathbotaccentwide", "combining right arrow below")
addSymbol(0x020F0, "asteraccent", "mathaccent", "combining asterisk above")
addSymbol(0x02102, "BbbC", "mathalpha", "/bbb c, open face c")
addSymbol(0x02107, "Eulerconst", "mathord", "euler constant")
addSymbol(0x0210A, "mscrg", "mathalpha", "/scr g, script letter g")
addSymbol(0x0210B, "mscrH", "mathalpha", "hamiltonian (script capital h)")
addSymbol(0x0210C, "mfrakH", "mathalpha", "/frak h, upper case h")
addSymbol(0x0210D, "BbbH", "mathalpha", "/bbb h, open face h")
addSymbol(0x0210E, "Planckconst", "mathord", "planck constant")
addSymbol(0x0210F, "hslash", "mathalpha", "/hslash - variant planck's over 2pi")
addSymbol(0x02110, "mscrI", "mathalpha", "/scr i, script letter i")
addSymbol(0x02111, "Im", "mathalpha", "imaginary part")
addSymbol(0x02112, "mscrL", "mathalpha", "lagrangian (script capital l)")
addSymbol(0x02113, "ell", "mathalpha", "cursive small l")
addSymbol(0x02115, "BbbN", "mathalpha", "/bbb n, open face n")
addSymbol(0x02118, "wp", "mathalpha", "weierstrass p")
addSymbol(0x02119, "BbbP", "mathalpha", "/bbb p, open face p")
addSymbol(0x0211A, "BbbQ", "mathalpha", "/bbb q, open face q")
addSymbol(0x0211B, "mscrR", "mathalpha", "/scr r, script letter r")
addSymbol(0x0211C, "Re", "mathalpha", "real part")
addSymbol(0x0211D, "BbbR", "mathalpha", "/bbb r, open face r")
addSymbol(0x02124, "BbbZ", "mathalpha", "/bbb z, open face z")
addSymbol(0x02127, "mho", "mathord", "conductance")
addSymbol(0x02128, "mfrakZ", "mathalpha", "/frak z, upper case z")
addSymbol(0x02129, "turnediota", "mathalpha", "turned iota")
addSymbol(0x0212B, "Angstrom", "mathalpha", "angstrom capital a, ring")
addSymbol(0x0212C, "mscrB", "mathalpha", "bernoulli function (script capital b)")
addSymbol(0x0212D, "mfrakC", "mathalpha", "black-letter capital c")
addSymbol(0x0212F, "mscre", "mathalpha", "/scr e, script letter e")
addSymbol(0x02130, "mscrE", "mathalpha", "/scr e, script letter e")
addSymbol(0x02131, "mscrF", "mathalpha", "/scr f, script letter f")
addSymbol(0x02132, "Finv", "mathord", "turned capital f")
addSymbol(0x02133, "mscrM", "mathalpha", "physics m-matrix (script capital m)")
addSymbol(0x02134, "mscro", "mathalpha", "order of (script small o)")
addSymbol(0x02135, "aleph", "mathalpha", "aleph, hebrew")
addSymbol(0x02136, "beth", "mathalpha", "beth, hebrew")
addSymbol(0x02137, "gimel", "mathalpha", "gimel, hebrew")
addSymbol(0x02138, "daleth", "mathalpha", "daleth, hebrew")
addSymbol(0x0213C, "Bbbpi", "mathord", "double-struck small pi")
addSymbol(0x0213D, "Bbbgamma", "mathalpha", "double-struck small gamma")
addSymbol(0x0213E, "BbbGamma", "mathalpha", "double-struck capital gamma")
addSymbol(0x0213F, "BbbPi", "mathalpha", "double-struck capital pi")
addSymbol(0x02140, "Bbbsum", "mathop", "double-struck n-ary summation")
addSymbol(0x02141, "Game", "mathord", "turned sans-serif capital g")
addSymbol(0x02142, "sansLturned", "mathord", "turned sans-serif capital l")
addSymbol(0x02143, "sansLmirrored", "mathord", "reversed sans-serif capital l")
addSymbol(0x02144, "Yup", "mathord", "turned sans-serif capital y")
addSymbol(0x02145, "mitBbbD", "mathord", "double-struck italic capital d")
addSymbol(0x02146, "mitBbbd", "mathord", "double-struck italic small d")
addSymbol(0x02147, "mitBbbe", "mathord", "double-struck italic small e")
addSymbol(0x02148, "mitBbbi", "mathord", "double-struck italic small i")
addSymbol(0x02149, "mitBbbj", "mathord", "double-struck italic small j")
addSymbol(0x0214A, "PropertyLine", "mathord", "property line")
addSymbol(0x0214B, "upand", "mathbin", "turned ampersand")
addSymbol(0x02190, "leftarrow", "mathrel", "/leftarrow /gets a: leftward arrow")
addSymbol(0x02191, "uparrow", "mathrel", "upward arrow")
addSymbol(0x02192, "rightarrow", "mathrel", "/rightarrow /to a: rightward arrow")
addSymbol(0x02193, "downarrow", "mathrel", "downward arrow")
addSymbol(0x02194, "leftrightarrow", "mathrel", "left and right arrow")
addSymbol(0x02195, "updownarrow", "mathrel", "up and down arrow")
addSymbol(0x02196, "nwarrow", "mathrel", "nw pointing arrow")
addSymbol(0x02197, "nearrow", "mathrel", "ne pointing arrow")
addSymbol(0x02198, "searrow", "mathrel", "se pointing arrow")
addSymbol(0x02199, "swarrow", "mathrel", "sw pointing arrow")
addSymbol(0x0219A, "nleftarrow", "mathrel", "not left arrow")
addSymbol(0x0219B, "nrightarrow", "mathrel", "not right arrow")
addSymbol(0x0219C, "leftwavearrow", "mathrel", "left arrow-wavy")
addSymbol(0x0219D, "rightwavearrow", "mathrel", "right arrow-wavy")
addSymbol(0x0219E, "twoheadleftarrow", "mathrel", "left two-headed arrow")
addSymbol(0x0219F, "twoheaduparrow", "mathrel", "up two-headed arrow")
addSymbol(0x021A0, "twoheadrightarrow", "mathrel", "right two-headed arrow")
addSymbol(0x021A1, "twoheaddownarrow", "mathrel", "down two-headed arrow")
addSymbol(0x021A2, "leftarrowtail", "mathrel", "left arrow-tailed")
addSymbol(0x021A3, "rightarrowtail", "mathrel", "right arrow-tailed")
addSymbol(0x021A4, "mapsfrom", "mathrel", "maps to, leftward")
addSymbol(0x021A5, "mapsup", "mathrel", "maps to, upward")
addSymbol(0x021A6, "mapsto", "mathrel", "maps to, rightward")
addSymbol(0x021A7, "mapsdown", "mathrel", "maps to, downward")
addSymbol(0x021A8, "updownarrowbar", "mathord", "up down arrow with base (perpendicular)")
addSymbol(0x021A9, "hookleftarrow", "mathrel", "left arrow-hooked")
addSymbol(0x021AA, "hookrightarrow", "mathrel", "right arrow-hooked")
addSymbol(0x021AB, "looparrowleft", "mathrel", "left arrow-looped")
addSymbol(0x021AC, "looparrowright", "mathrel", "right arrow-looped")
addSymbol(0x021AD, "leftrightsquigarrow", "mathrel", "left and right arr-wavy")
addSymbol(0x021AE, "nleftrightarrow", "mathrel", "not left and right arrow")
addSymbol(0x021AF, "downzigzagarrow", "mathrel", "downwards zigzag arrow")
addSymbol(0x021B0, "Lsh", "mathrel", "/lsh a:")
addSymbol(0x021B1, "Rsh", "mathrel", "/rsh a:")
addSymbol(0x021B2, "Ldsh", "mathrel", "left down angled arrow")
addSymbol(0x021B3, "Rdsh", "mathrel", "right down angled arrow")
addSymbol(0x021B4, "linefeed", "mathord", "rightwards arrow with corner downwards")
addSymbol(0x021B5, "carriagereturn", "mathord", "downwards arrow with corner leftward = carriage return")
addSymbol(0x021B6, "curvearrowleft", "mathrel", "left curved arrow")
addSymbol(0x021B7, "curvearrowright", "mathrel", "right curved arrow")
addSymbol(0x021B8, "barovernorthwestarrow", "mathord", "north west arrow to long bar")
addSymbol(0x021B9, "barleftarrowrightarrowbar", "mathord", "leftwards arrow to bar over rightwards arrow to bar")
addSymbol(0x021BA, "acwopencirclearrow", "mathord", "anticlockwise open circle arrow")
addSymbol(0x021BB, "cwopencirclearrow", "mathord", "clockwise open circle arrow")
addSymbol(0x021BC, "leftharpoonup", "mathrel", "left harpoon-up")
addSymbol(0x021BD, "leftharpoondown", "mathrel", "left harpoon-down")
addSymbol(0x021BE, "upharpoonright", "mathrel", "/upharpoonright /restriction a: up harpoon-right")
addSymbol(0x021BF, "upharpoonleft", "mathrel", "up harpoon-left")
addSymbol(0x021C0, "rightharpoonup", "mathrel", "right harpoon-up")
addSymbol(0x021C1, "rightharpoondown", "mathrel", "right harpoon-down")
addSymbol(0x021C2, "downharpoonright", "mathrel", "down harpoon-right")
addSymbol(0x021C3, "downharpoonleft", "mathrel", "down harpoon-left")
addSymbol(0x021C4, "rightleftarrows", "mathrel", "right arrow over left arrow")
addSymbol(0x021C5, "updownarrows", "mathrel", "up arrow, down arrow")
addSymbol(0x021C6, "leftrightarrows", "mathrel", "left arrow over right arrow")
addSymbol(0x021C7, "leftleftarrows", "mathrel", "two left arrows")
addSymbol(0x021C8, "upuparrows", "mathrel", "two up arrows")
addSymbol(0x021C9, "rightrightarrows", "mathrel", "two right arrows")
addSymbol(0x021CA, "downdownarrows", "mathrel", "two down arrows")
addSymbol(0x021CB, "leftrightharpoons", "mathrel", "left harpoon over right")
addSymbol(0x021CC, "rightleftharpoons", "mathrel", "right harpoon over left")
addSymbol(0x021CD, "nLeftarrow", "mathrel", "not implied by")
addSymbol(0x021CE, "nLeftrightarrow", "mathrel", "not left and right double arrows")
addSymbol(0x021CF, "nRightarrow", "mathrel", "not implies")
addSymbol(0x021D0, "Leftarrow", "mathrel", "is implied by")
addSymbol(0x021D1, "Uparrow", "mathrel", "up double arrow")
addSymbol(0x021D2, "Rightarrow", "mathrel", "implies")
addSymbol(0x021D3, "Downarrow", "mathrel", "down double arrow")
addSymbol(0x021D4, "Leftrightarrow", "mathrel", "left and right double arrow")
addSymbol(0x021D5, "Updownarrow", "mathrel", "up and down double arrow")
addSymbol(0x021D6, "Nwarrow", "mathrel", "nw pointing double arrow")
addSymbol(0x021D7, "Nearrow", "mathrel", "ne pointing double arrow")
addSymbol(0x021D8, "Searrow", "mathrel", "se pointing double arrow")
addSymbol(0x021D9, "Swarrow", "mathrel", "sw pointing double arrow")
addSymbol(0x021DA, "Lleftarrow", "mathrel", "left triple arrow")
addSymbol(0x021DB, "Rrightarrow", "mathrel", "right triple arrow")
addSymbol(0x021DC, "leftsquigarrow", "mathrel", "leftwards squiggle arrow")
addSymbol(0x021DD, "rightsquigarrow", "mathrel", "rightwards squiggle arrow")
addSymbol(0x021DE, "nHuparrow", "mathord", "upwards arrow with double stroke")
addSymbol(0x021DF, "nHdownarrow", "mathord", "downwards arrow with double stroke")
addSymbol(0x021E0, "leftdasharrow", "mathord", "leftwards dashed arrow")
addSymbol(0x021E1, "updasharrow", "mathord", "upwards dashed arrow")
addSymbol(0x021E2, "rightdasharrow", "mathord", "rightwards dashed arrow")
addSymbol(0x021E3, "downdasharrow", "mathord", "downwards dashed arrow")
addSymbol(0x021E4, "barleftarrow", "mathrel", "leftwards arrow to bar")
addSymbol(0x021E5, "rightarrowbar", "mathrel", "rightwards arrow to bar")
addSymbol(0x021E6, "leftwhitearrow", "mathord", "leftwards white arrow")
addSymbol(0x021E7, "upwhitearrow", "mathord", "upwards white arrow")
addSymbol(0x021E8, "rightwhitearrow", "mathord", "rightwards white arrow")
addSymbol(0x021E9, "downwhitearrow", "mathord", "downwards white arrow")
addSymbol(0x021EA, "whitearrowupfrombar", "mathord", "upwards white arrow from bar")
addSymbol(0x021F4, "circleonrightarrow", "mathrel", "right arrow with small circle")
addSymbol(0x021F5, "downuparrows", "mathrel", "downwards arrow leftwards of upwards arrow")
addSymbol(0x021F6, "rightthreearrows", "mathrel", "three rightwards arrows")
addSymbol(0x021F7, "nvleftarrow", "mathrel", "leftwards arrow with vertical stroke")
addSymbol(0x021F8, "nvrightarrow", "mathrel", "rightwards arrow with vertical stroke")
addSymbol(0x021F9, "nvleftrightarrow", "mathrel", "left right arrow with vertical stroke")
addSymbol(0x021FA, "nVleftarrow", "mathrel", "leftwards arrow with double vertical stroke")
addSymbol(0x021FB, "nVrightarrow", "mathrel", "rightwards arrow with double vertical stroke")
addSymbol(0x021FC, "nVleftrightarrow", "mathrel", "left right arrow with double vertical stroke")
addSymbol(0x021FD, "leftarrowtriangle", "mathrel", "leftwards open-headed arrow")
addSymbol(0x021FE, "rightarrowtriangle", "mathrel", "rightwards open-headed arrow")
addSymbol(0x021FF, "leftrightarrowtriangle", "mathrel", "left right open-headed arrow")
addSymbol(0x02200, "forall", "mathord", "for all")
addSymbol(0x02201, "complement", "mathord", "complement sign")
addSymbol(0x02202, "partial", "mathalpha", "partial differential")
addSymbol(0x02203, "exists", "mathord", "at least one exists")
addSymbol(0x02204, "nexists", "mathord", "negated exists")
addSymbol(0x02205, "varnothing", "mathord", "circle, slash")
addSymbol(0x02206, "increment", "mathord", "laplacian (delta; nabla^2)")
addSymbol(0x02207, "nabla", "mathalpha", "nabla, del, hamilton operator")
addSymbol(0x02208, "in", "mathrel", "set membership, variant")
addSymbol(0x02209, "notin", "mathrel", "negated set membership")
addSymbol(0x0220A, "smallin", "mathrel", "set membership (small set membership)")
addSymbol(0x0220B, "ni", "mathrel", "contains, variant")
addSymbol(0x0220C, "nni", "mathrel", "negated contains, variant")
addSymbol(0x0220D, "smallni", "mathrel", "/ni /owns r: contains (small contains as member)")
addSymbol(0x0220E, "QED", "mathord", "end of proof")
addSymbol(0x0220F, "prod", "mathop", "product operator")
addSymbol(0x02210, "coprod", "mathop", "coproduct operator")
addSymbol(0x02211, "sum", "mathop", "summation operator")
addSymbol(0x02212, "minus", "mathbin", "minus sign")
addSymbol(0x02213, "mp", "mathbin", "minus-or-plus sign")
addSymbol(0x02214, "dotplus", "mathbin", "plus sign, dot above")
addSymbol(0x02215, "divslash", "mathbin", "division slash")
addSymbol(0x02216, "smallsetminus", "mathbin", "small set minus (cf. reverse solidus)")
addSymbol(0x02217, "ast", "mathbin", "centered asterisk")
addSymbol(0x02218, "vysmwhtcircle", "mathbin", "composite function (small circle)")
addSymbol(0x02219, "vysmblkcircle", "mathbin", "bullet operator")
addSymbol(0x0221A, "sqrt", "mathopen", "radical")
addSymbol(0x0221A, "surd", "mathord", "radical")
addSymbol(0x0221B, "cuberoot", "mathopen", "cube root")
addSymbol(0x0221C, "fourthroot", "mathopen", "fourth root")
addSymbol(0x0221D, "propto", "mathrel", "is proportional to")
addSymbol(0x0221E, "infty", "mathord", "infinity")
addSymbol(0x0221F, "rightangle", "mathord", "right (90 degree) angle")
addSymbol(0x02220, "angle", "mathord", "angle")
addSymbol(0x02221, "measuredangle", "mathord", "angle-measured")
addSymbol(0x02222, "sphericalangle", "mathord", "angle-spherical")
addSymbol(0x02223, "mid", "mathrel", "/mid r:")
addSymbol(0x02224, "nmid", "mathrel", "negated mid")
addSymbol(0x02225, "parallel", "mathrel", "parallel")
addSymbol(0x02226, "nparallel", "mathrel", "not parallel")
addSymbol(0x02227, "wedge", "mathbin", "/wedge /land b: logical and")
addSymbol(0x02228, "vee", "mathbin", "/vee /lor b: logical or")
addSymbol(0x02229, "cap", "mathbin", "intersection")
addSymbol(0x0222A, "cup", "mathbin", "union or logical sum")
addSymbol(0x0222B, "int", "mathop", "integral operator")
addSymbol(0x0222C, "iint", "mathop", "double integral operator")
addSymbol(0x0222D, "iiint", "mathop", "triple integral operator")
addSymbol(0x0222E, "oint", "mathop", "contour integral operator")
addSymbol(0x0222F, "oiint", "mathop", "double contour integral operator")
addSymbol(0x02230, "oiiint", "mathop", "triple contour integral operator")
addSymbol(0x02231, "intclockwise", "mathop", "clockwise integral")
addSymbol(0x02232, "varointclockwise", "mathop", "contour integral, clockwise")
addSymbol(0x02233, "ointctrclockwise", "mathop", "contour integral, anticlockwise")
addSymbol(0x02234, "therefore", "mathord", "therefore")
addSymbol(0x02235, "because", "mathord", "because")
addSymbol(0x02236, "mathratio", "mathrel", "ratio")
addSymbol(0x02237, "Colon", "mathrel", "two colons")
addSymbol(0x02238, "dotminus", "mathbin", "minus sign, dot above")
addSymbol(0x02239, "dashcolon", "mathrel", "excess (-:)")
addSymbol(0x0223A, "dotsminusdots", "mathrel", "minus with four dots, geometric properties")
addSymbol(0x0223B, "kernelcontraction", "mathrel", "homothetic")
addSymbol(0x0223C, "sim", "mathrel", "similar")
addSymbol(0x0223D, "backsim", "mathrel", "reverse similar")
addSymbol(0x0223E, "invlazys", "mathbin", "most positive [inverted lazy s]")
addSymbol(0x0223F, "sinewave", "mathord", "sine wave")
addSymbol(0x02240, "wr", "mathbin", "wreath product")
addSymbol(0x02241, "nsim", "mathrel", "not similar")
addSymbol(0x02242, "eqsim", "mathrel", "equals, similar")
addSymbol(0x02243, "simeq", "mathrel", "similar, equals")
addSymbol(0x02244, "nsime", "mathrel", "not similar, equals")
addSymbol(0x02243, "sime", "mathrel", "similar, equals (alias)")
addSymbol(0x02244, "nsimeq", "mathrel", "not similar, equals (alias)")
addSymbol(0x02245, "cong", "mathrel", "congruent with")
addSymbol(0x02246, "simneqq", "mathrel", "similar, not equals [vert only for 9573 entity]")
addSymbol(0x02247, "ncong", "mathrel", "not congruent with")
addSymbol(0x02248, "approx", "mathrel", "approximate")
addSymbol(0x02249, "napprox", "mathrel", "not approximate")
addSymbol(0x0224A, "approxeq", "mathrel", "approximate, equals")
addSymbol(0x0224B, "approxident", "mathrel", "approximately identical to")
addSymbol(0x0224C, "backcong", "mathrel", "all equal to")
addSymbol(0x0224D, "asymp", "mathrel", "asymptotically equal to")
addSymbol(0x0224E, "Bumpeq", "mathrel", "bumpy equals")
addSymbol(0x0224F, "bumpeq", "mathrel", "bumpy equals, equals")
addSymbol(0x02250, "doteq", "mathrel", "equals, single dot above")
addSymbol(0x02251, "Doteq", "mathrel", "/doteqdot /doteq r: equals, even dots")
addSymbol(0x02252, "fallingdotseq", "mathrel", "equals, falling dots")
addSymbol(0x02253, "risingdotseq", "mathrel", "equals, rising dots")
addSymbol(0x02254, "coloneq", "mathrel", "colon, equals")
addSymbol(0x02255, "eqcolon", "mathrel", "equals, colon")
addSymbol(0x02256, "eqcirc", "mathrel", "circle on equals sign")
addSymbol(0x02257, "circeq", "mathrel", "circle, equals")
addSymbol(0x02258, "arceq", "mathrel", "arc, equals; corresponds to")
addSymbol(0x02259, "wedgeq", "mathrel", "corresponds to (wedge, equals)")
addSymbol(0x0225A, "veeeq", "mathrel", "logical or, equals")
addSymbol(0x0225B, "stareq", "mathrel", "star equals")
addSymbol(0x0225C, "triangleq", "mathrel", "triangle, equals")
addSymbol(0x0225D, "eqdef", "mathrel", "equals by definition")
addSymbol(0x0225E, "measeq", "mathrel", "measured by (m over equals)")
addSymbol(0x0225F, "questeq", "mathrel", "equal with questionmark")
addSymbol(0x02260, "ne", "mathrel", "/ne /neq r: not equal")
addSymbol(0x02261, "equiv", "mathrel", "identical with")
addSymbol(0x02262, "nequiv", "mathrel", "not identical with")
addSymbol(0x02263, "Equiv", "mathrel", "strict equivalence (4 lines)")
addSymbol(0x02264, "leq", "mathrel", "/leq /le r: less-than-or-equal")
addSymbol(0x02265, "geq", "mathrel", "/geq /ge r: greater-than-or-equal")
addSymbol(0x02266, "leqq", "mathrel", "less, double equals")
addSymbol(0x02267, "geqq", "mathrel", "greater, double equals")
addSymbol(0x02268, "lneqq", "mathrel", "less, not double equals")
addSymbol(0x02269, "gneqq", "mathrel", "greater, not double equals")
addSymbol(0x0226A, "ll", "mathrel", "much less than, type 2")
addSymbol(0x0226B, "gg", "mathrel", "much greater than, type 2")
addSymbol(0x0226C, "between", "mathrel", "between")
addSymbol(0x0226D, "nasymp", "mathrel", "not asymptotically equal to")
addSymbol(0x0226E, "nless", "mathrel", "not less-than")
addSymbol(0x0226F, "ngtr", "mathrel", "not greater-than")
addSymbol(0x02270, "nleq", "mathrel", "not less-than-or-equal")
addSymbol(0x02271, "ngeq", "mathrel", "not greater-than-or-equal")
addSymbol(0x02272, "lesssim", "mathrel", "less, similar")
addSymbol(0x02273, "gtrsim", "mathrel", "greater, similar")
addSymbol(0x02274, "nlesssim", "mathrel", "not less, similar")
addSymbol(0x02275, "ngtrsim", "mathrel", "not greater, similar")
addSymbol(0x02276, "lessgtr", "mathrel", "less, greater")
addSymbol(0x02277, "gtrless", "mathrel", "greater, less")
addSymbol(0x02278, "nlessgtr", "mathrel", "not less, greater")
addSymbol(0x02279, "ngtrless", "mathrel", "not greater, less")
addSymbol(0x0227A, "prec", "mathrel", "precedes")
addSymbol(0x0227B, "succ", "mathrel", "succeeds")
addSymbol(0x0227C, "preccurlyeq", "mathrel", "precedes, curly equals")
addSymbol(0x0227D, "succcurlyeq", "mathrel", "succeeds, curly equals")
addSymbol(0x0227E, "precsim", "mathrel", "precedes, similar")
addSymbol(0x0227F, "succsim", "mathrel", "succeeds, similar")
addSymbol(0x02280, "nprec", "mathrel", "not precedes")
addSymbol(0x02281, "nsucc", "mathrel", "not succeeds")
addSymbol(0x02282, "subset", "mathrel", "subset or is implied by")
addSymbol(0x02283, "supset", "mathrel", "superset or implies")
addSymbol(0x02284, "nsubset", "mathrel", "not subset, variant [slash negation]")
addSymbol(0x02285, "nsupset", "mathrel", "not superset, variant [slash negation]")
addSymbol(0x02286, "subseteq", "mathrel", "subset, equals")
addSymbol(0x02287, "supseteq", "mathrel", "superset, equals")
addSymbol(0x02288, "nsubseteq", "mathrel", "not subset, equals")
addSymbol(0x02289, "nsupseteq", "mathrel", "not superset, equals")
addSymbol(0x0228A, "subsetneq", "mathrel", "subset, not equals")
addSymbol(0x0228B, "supsetneq", "mathrel", "superset, not equals")
addSymbol(0x0228C, "cupleftarrow", "mathbin", "multiset")
addSymbol(0x0228D, "cupdot", "mathbin", "union, with dot")
addSymbol(0x0228E, "uplus", "mathbin", "plus sign in union")
addSymbol(0x0228F, "sqsubset", "mathrel", "square subset")
addSymbol(0x02290, "sqsupset", "mathrel", "square superset")
addSymbol(0x02291, "sqsubseteq", "mathrel", "square subset, equals")
addSymbol(0x02292, "sqsupseteq", "mathrel", "square superset, equals")
addSymbol(0x02293, "sqcap", "mathbin", "square intersection")
addSymbol(0x02294, "sqcup", "mathbin", "square union")
addSymbol(0x02295, "oplus", "mathbin", "plus sign in circle")
addSymbol(0x02296, "ominus", "mathbin", "minus sign in circle")
addSymbol(0x02297, "otimes", "mathbin", "multiply sign in circle")
addSymbol(0x02298, "oslash", "mathbin", "solidus in circle")
addSymbol(0x02299, "odot", "mathbin", "middle dot in circle")
addSymbol(0x0229A, "circledcirc", "mathbin", "small circle in circle")
addSymbol(0x0229B, "circledast", "mathbin", "asterisk in circle")
addSymbol(0x0229C, "circledequal", "mathbin", "equal in circle")
addSymbol(0x0229D, "circleddash", "mathbin", "hyphen in circle")
addSymbol(0x0229E, "boxplus", "mathbin", "plus sign in box")
addSymbol(0x0229F, "boxminus", "mathbin", "minus sign in box")
addSymbol(0x022A0, "boxtimes", "mathbin", "multiply sign in box")
addSymbol(0x022A1, "boxdot", "mathbin", "/dotsquare /boxdot b: small dot in box")
addSymbol(0x022A2, "vdash", "mathrel", "vertical, dash")
addSymbol(0x022A3, "dashv", "mathrel", "dash, vertical")
addSymbol(0x022A4, "top", "mathord", "top")
addSymbol(0x022A5, "bot", "mathord", "bottom")
addSymbol(0x022A6, "assert", "mathrel", "assertion (vertical, short dash)")
addSymbol(0x022A7, "models", "mathrel", "models (vertical, short double dash)")
addSymbol(0x022A8, "vDash", "mathrel", "vertical, double dash")
addSymbol(0x022A9, "Vdash", "mathrel", "double vertical, dash")
addSymbol(0x022AA, "Vvdash", "mathrel", "triple vertical, dash")
addSymbol(0x022AB, "VDash", "mathrel", "double vert, double dash")
addSymbol(0x022AC, "nvdash", "mathrel", "not vertical, dash")
addSymbol(0x022AD, "nvDash", "mathrel", "not vertical, double dash")
addSymbol(0x022AE, "nVdash", "mathrel", "not double vertical, dash")
addSymbol(0x022AF, "nVDash", "mathrel", "not double vert, double dash")
addSymbol(0x022B0, "prurel", "mathrel", "element precedes under relation")
addSymbol(0x022B1, "scurel", "mathrel", "succeeds under relation")
addSymbol(0x022B2, "vartriangleleft", "mathrel", "left triangle, open, variant")
addSymbol(0x022B3, "vartriangleright", "mathrel", "right triangle, open, variant")
addSymbol(0x022B4, "trianglelefteq", "mathrel", "left triangle, equals")
addSymbol(0x022B5, "trianglerighteq", "mathrel", "right triangle, equals")
addSymbol(0x022B6, "origof", "mathrel", "original of")
addSymbol(0x022B7, "imageof", "mathrel", "image of")
addSymbol(0x022B8, "multimap", "mathrel", "/multimap a:")
addSymbol(0x022B9, "hermitmatrix", "mathord", "hermitian conjugate matrix")
addSymbol(0x022BA, "intercal", "mathbin", "intercal")
addSymbol(0x022BB, "veebar", "mathbin", "logical or, bar below (large vee); exclusive disjunction")
addSymbol(0x022BC, "barwedge", "mathbin", "bar, wedge (large wedge)")
addSymbol(0x022BD, "barvee", "mathbin", "bar, vee (large vee)")
addSymbol(0x022BE, "measuredrightangle", "mathord", "right angle-measured [with arc]")
addSymbol(0x022BF, "varlrtriangle", "mathord", "right triangle")
addSymbol(0x022C0, "bigwedge", "mathop", "logical and operator")
addSymbol(0x022C1, "bigvee", "mathop", "logical or operator")
addSymbol(0x022C2, "bigcap", "mathop", "intersection operator")
addSymbol(0x022C3, "bigcup", "mathop", "union operator")
addSymbol(0x022C4, "smwhtdiamond", "mathbin", "white diamond")
addSymbol(0x022C5, "cdot", "mathbin", "small middle dot")
addSymbol(0x022C6, "star", "mathbin", "small star, filled, low")
addSymbol(0x022C7, "divideontimes", "mathbin", "division on times")
addSymbol(0x022C8, "bowtie", "mathrel", "bowtie")
addSymbol(0x022C9, "ltimes", "mathbin", "times sign, left closed")
addSymbol(0x022CA, "rtimes", "mathbin", "times sign, right closed")
addSymbol(0x022CB, "leftthreetimes", "mathbin", "left semidirect product")
addSymbol(0x022CC, "rightthreetimes", "mathbin", "right semidirect product")
addSymbol(0x022CD, "backsimeq", "mathrel", "reverse similar, equals")
addSymbol(0x022CE, "curlyvee", "mathbin", "curly logical or")
addSymbol(0x022CF, "curlywedge", "mathbin", "curly logical and")
addSymbol(0x022D0, "Subset", "mathrel", "double subset")
addSymbol(0x022D1, "Supset", "mathrel", "double superset")
addSymbol(0x022D2, "Cap", "mathbin", "/cap /doublecap b: double intersection")
addSymbol(0x022D3, "Cup", "mathbin", "/cup /doublecup b: double union")
addSymbol(0x022D4, "pitchfork", "mathrel", "pitchfork")
addSymbol(0x022D5, "equalparallel", "mathrel", "parallel, equal; equal or parallel")
addSymbol(0x022D6, "lessdot", "mathrel", "less than, with dot")
addSymbol(0x022D7, "gtrdot", "mathrel", "greater than, with dot")
addSymbol(0x022D8, "lll", "mathrel", "/ll /lll /llless r: triple less-than")
addSymbol(0x022D9, "ggg", "mathrel", "/ggg /gg /gggtr r: triple greater-than")
addSymbol(0x022DA, "lesseqgtr", "mathrel", "less, equals, greater")
addSymbol(0x022DB, "gtreqless", "mathrel", "greater, equals, less")
addSymbol(0x022DC, "eqless", "mathrel", "equal-or-less")
addSymbol(0x022DD, "eqgtr", "mathrel", "equal-or-greater")
addSymbol(0x022DE, "curlyeqprec", "mathrel", "curly equals, precedes")
addSymbol(0x022DF, "curlyeqsucc", "mathrel", "curly equals, succeeds")
addSymbol(0x022E0, "npreccurlyeq", "mathrel", "not precedes, curly equals")
addSymbol(0x022E1, "nsucccurlyeq", "mathrel", "not succeeds, curly equals")
addSymbol(0x022E2, "nsqsubseteq", "mathrel", "not, square subset, equals")
addSymbol(0x022E3, "nsqsupseteq", "mathrel", "not, square superset, equals")
addSymbol(0x022E4, "sqsubsetneq", "mathrel", "square subset, not equals")
addSymbol(0x022E5, "sqsupsetneq", "mathrel", "square superset, not equals")
addSymbol(0x022E6, "lnsim", "mathrel", "less, not similar")
addSymbol(0x022E7, "gnsim", "mathrel", "greater, not similar")
addSymbol(0x022E8, "precnsim", "mathrel", "precedes, not similar")
addSymbol(0x022E9, "succnsim", "mathrel", "succeeds, not similar")
addSymbol(0x022EA, "nvartriangleleft", "mathrel", "not left triangle")
addSymbol(0x022EB, "nvartriangleright", "mathrel", "not right triangle")
addSymbol(0x022EC, "ntrianglelefteq", "mathrel", "not left triangle, equals")
addSymbol(0x022ED, "ntrianglerighteq", "mathrel", "not right triangle, equals")
addSymbol(0x022EE, "vdots", "mathrel", "vertical ellipsis")
addSymbol(0x022EF, "unicodecdots", "mathord", "three dots, centered")
addSymbol(0x022F0, "adots", "mathrel", "three dots, ascending")
addSymbol(0x022F1, "ddots", "mathrel", "three dots, descending")
addSymbol(0x022F2, "disin", "mathrel", "element of with long horizontal stroke")
addSymbol(0x022F3, "varisins", "mathrel", "element of with vertical bar at end of horizontal stroke")
addSymbol(0x022F4, "isins", "mathrel", "small element of with vertical bar at end of horizontal stroke")
addSymbol(0x022F5, "isindot", "mathrel", "element of with dot above")
addSymbol(0x022F6, "varisinobar", "mathrel", "element of with overbar")
addSymbol(0x022F7, "isinobar", "mathrel", "small element of with overbar")
addSymbol(0x022F8, "isinvb", "mathrel", "element of with underbar")
addSymbol(0x022F9, "isinE", "mathrel", "element of with two horizontal strokes")
addSymbol(0x022FA, "nisd", "mathrel", "contains with long horizontal stroke")
addSymbol(0x022FB, "varnis", "mathrel", "contains with vertical bar at end of horizontal stroke")
addSymbol(0x022FC, "nis", "mathrel", "small contains with vertical bar at end of horizontal stroke")
addSymbol(0x022FD, "varniobar", "mathrel", "contains with overbar")
addSymbol(0x022FE, "niobar", "mathrel", "small contains with overbar")
addSymbol(0x022FF, "bagmember", "mathrel", "z notation bag membership")
addSymbol(0x02300, "diameter", "mathord", "diameter sign")
addSymbol(0x02302, "house", "mathord", "house")
addSymbol(0x02305, "varbarwedge", "mathbin", "/barwedge b: logical and, bar above [projective (bar over small wedge)]")
addSymbol(
   0x02306,
   "vardoublebarwedge",
   "mathbin",
   "/doublebarwedge b: logical and, double bar above [perspective (double bar over small wedge)]"
)
addSymbol(0x02308, "lceil", "mathopen", "left ceiling")
addSymbol(0x02309, "rceil", "mathclose", "right ceiling")
addSymbol(0x0230A, "lfloor", "mathopen", "left floor")
addSymbol(0x0230B, "rfloor", "mathclose", "right floor")
addSymbol(0x02310, "invnot", "mathord", "reverse not")
addSymbol(0x02311, "sqlozenge", "mathord", "square lozenge")
addSymbol(0x02312, "profline", "mathord", "profile of a line")
addSymbol(0x02313, "profsurf", "mathord", "profile of a surface")
addSymbol(0x02317, "viewdata", "mathord", "viewdata square")
addSymbol(0x02319, "turnednot", "mathord", "turned not sign")
addSymbol(0x0231C, "ulcorner", "mathopen", "upper left corner")
addSymbol(0x0231D, "urcorner", "mathclose", "upper right corner")
addSymbol(0x0231E, "llcorner", "mathopen", "lower left corner")
addSymbol(0x0231F, "lrcorner", "mathclose", "lower right corner")
addSymbol(0x02320, "inttop", "mathord", "top half integral")
addSymbol(0x02321, "intbottom", "mathord", "bottom half integral")
addSymbol(0x02322, "frown", "mathrel", "down curve")
addSymbol(0x02323, "smile", "mathrel", "up curve")
addSymbol(0x0232C, "varhexagonlrbonds", "mathord", "six carbon ring, corner down, double bonds lower right etc")
addSymbol(0x02332, "conictaper", "mathord", "conical taper ")
addSymbol(0x02336, "topbot", "mathord", "top and bottom")
addSymbol(0x0233D, "obar", "mathbin", "circle with vertical bar")
addSymbol(0x0233F, "APLnotslash", "mathrel", "solidus, bar through (apl functional symbol slash bar)")
addSymbol(0x02340, "APLnotbackslash", "mathord", "apl functional symbol backslash bar")
addSymbol(0x02353, "APLboxupcaret", "mathord", "boxed up caret")
addSymbol(0x02370, "APLboxquestion", "mathord", "boxed question mark")
addSymbol(0x0237C, "rangledownzigzagarrow", "mathord", "right angle with downwards zigzag arrow")
addSymbol(0x02394, "hexagon", "mathord", "horizontal benzene ring [hexagon flat open]")
addSymbol(0x0239B, "lparenuend", "mathord", "left parenthesis upper hook")
addSymbol(0x0239C, "lparenextender", "mathord", "left parenthesis extension")
addSymbol(0x0239D, "lparenlend", "mathord", "left parenthesis lower hook")
addSymbol(0x0239E, "rparenuend", "mathord", "right parenthesis upper hook")
addSymbol(0x0239F, "rparenextender", "mathord", "right parenthesis extension")
addSymbol(0x023A0, "rparenlend", "mathord", "right parenthesis lower hook")
addSymbol(0x023A1, "lbrackuend", "mathord", "left square bracket upper corner")
addSymbol(0x023A2, "lbrackextender", "mathord", "left square bracket extension")
addSymbol(0x023A3, "lbracklend", "mathord", "left square bracket lower corner")
addSymbol(0x023A4, "rbrackuend", "mathord", "right square bracket upper corner")
addSymbol(0x023A5, "rbrackextender", "mathord", "right square bracket extension")
addSymbol(0x023A6, "rbracklend", "mathord", "right square bracket lower corner")
addSymbol(0x023A7, "lbraceuend", "mathord", "left curly bracket upper hook")
addSymbol(0x023A8, "lbracemid", "mathord", "left curly bracket middle piece")
addSymbol(0x023A9, "lbracelend", "mathord", "left curly bracket lower hook")
addSymbol(0x023AA, "vbraceextender", "mathord", "curly bracket extension")
addSymbol(0x023AB, "rbraceuend", "mathord", "right curly bracket upper hook")
addSymbol(0x023AC, "rbracemid", "mathord", "right curly bracket middle piece")
addSymbol(0x023AD, "rbracelend", "mathord", "right curly bracket lower hook")
addSymbol(0x023AE, "intextender", "mathord", "integral extension")
addSymbol(0x023AF, "harrowextender", "mathord", "horizontal line extension (used to extend arrows)")
addSymbol(0x023B0, "lmoustache", "mathopen", "upper left or lower right curly bracket section")
addSymbol(0x023B1, "rmoustache", "mathclose", "upper right or lower left curly bracket section")
addSymbol(0x023B2, "sumtop", "mathord", "summation top")
addSymbol(0x023B3, "sumbottom", "mathord", "summation bottom")
addSymbol(0x023B4, "overbracket", "mathover", "top square bracket")
addSymbol(0x023B5, "underbracket", "mathunder", "bottom square bracket")
addSymbol(0x023B6, "bbrktbrk", "mathord", "bottom square bracket over top square bracket")
addSymbol(0x023B7, "sqrtbottom", "mathord", "radical symbol bottom")
addSymbol(0x023B8, "lvboxline", "mathord", "left vertical box line")
addSymbol(0x023B9, "rvboxline", "mathord", "right vertical box line")
addSymbol(0x023CE, "varcarriagereturn", "mathord", "return symbol")
addSymbol(0x023DC, "overparen", "mathover", "top parenthesis (mathematical use)")
addSymbol(0x023DD, "underparen", "mathunder", "bottom parenthesis (mathematical use)")
addSymbol(0x023DE, "overbrace", "mathover", "top curly bracket (mathematical use)")
addSymbol(0x023DF, "underbrace", "mathunder", "bottom curly bracket (mathematical use)")
addSymbol(0x023E0, "obrbrak", "mathord", "top tortoise shell bracket (mathematical use)")
addSymbol(0x023E1, "ubrbrak", "mathord", "bottom tortoise shell bracket (mathematical use)")
addSymbol(0x023E2, "trapezium", "mathord", "white trapezium")
addSymbol(0x023E3, "benzenr", "mathord", "benzene ring with circle")
addSymbol(0x023E4, "strns", "mathord", "straightness")
addSymbol(0x023E5, "fltns", "mathord", "flatness")
addSymbol(0x023E6, "accurrent", "mathord", "ac current")
addSymbol(0x023E7, "elinters", "mathord", "electrical intersection")
addSymbol(0x02422, "blanksymbol", "mathord", "blank symbol")
addSymbol(0x02423, "mathvisiblespace", "mathord", "open box")
addSymbol(0x02506, "bdtriplevdash", "mathord", "doubly broken vert")
addSymbol(0x02580, "blockuphalf", "mathord", "upper half block")
addSymbol(0x02584, "blocklowhalf", "mathord", "lower half block")
addSymbol(0x02588, "blockfull", "mathord", "full block")
addSymbol(0x0258C, "blocklefthalf", "mathord", "left half block")
addSymbol(0x02590, "blockrighthalf", "mathord", "right half block")
addSymbol(0x02591, "blockqtrshaded", "mathord", "25% shaded block")
addSymbol(0x02592, "blockhalfshaded", "mathord", "50% shaded block")
addSymbol(0x02593, "blockthreeqtrshaded", "mathord", "75% shaded block")
addSymbol(0x025A0, "mdlgblksquare", "mathord", "square, filled")
addSymbol(0x025A1, "mdlgwhtsquare", "mathord", "square, open")
addSymbol(0x025A2, "squoval", "mathord", "white square with rounded corners")
addSymbol(0x025A3, "blackinwhitesquare", "mathord", "white square containing black small square")
addSymbol(0x025A4, "squarehfill", "mathord", "square, horizontal rule filled")
addSymbol(0x025A5, "squarevfill", "mathord", "square, vertical rule filled")
addSymbol(0x025A6, "squarehvfill", "mathord", "square with orthogonal crosshatch fill")
addSymbol(0x025A7, "squarenwsefill", "mathord", "square, nw-to-se rule filled")
addSymbol(0x025A8, "squareneswfill", "mathord", "square, ne-to-sw rule filled")
addSymbol(0x025A9, "squarecrossfill", "mathord", "square with diagonal crosshatch fill")
addSymbol(0x025AA, "smblksquare", "mathord", "/blacksquare - sq bullet, filled")
addSymbol(0x025AB, "smwhtsquare", "mathord", "white small square")
addSymbol(0x025AC, "hrectangleblack", "mathord", "black rectangle")
addSymbol(0x025AD, "hrectangle", "mathord", "horizontal rectangle, open")
addSymbol(0x025AE, "vrectangleblack", "mathord", "black vertical rectangle")
addSymbol(0x025AF, "vrectangle", "mathord", "rectangle, white (vertical)")
addSymbol(0x025B0, "parallelogramblack", "mathord", "black parallelogram")
addSymbol(0x025B1, "parallelogram", "mathord", "parallelogram, open")
addSymbol(0x025B2, "bigblacktriangleup", "mathord", "black up-pointing triangle")
addSymbol(0x025B3, "bigtriangleup", "mathbin", "big up triangle, open")
addSymbol(0x025B4, "blacktriangle", "mathord", "up triangle, filled")
addSymbol(0x025B5, "vartriangle", "mathrel", "/triangle - up triangle, open")
addSymbol(0x025B6, "blacktriangleright", "mathord", "(large) right triangle, filled")
addSymbol(0x025B7, "triangleright", "mathbin", "(large) right triangle, open; z notation range restriction")
addSymbol(0x025B8, "smallblacktriangleright", "mathord", "right triangle, filled")
addSymbol(0x025B9, "smalltriangleright", "mathord", "right triangle, open")
addSymbol(0x025BA, "blackpointerright", "mathord", "black right-pointing pointer")
addSymbol(0x025BB, "whitepointerright", "mathord", "white right-pointing pointer")
addSymbol(0x025BC, "bigblacktriangledown", "mathord", "big down triangle, filled")
addSymbol(0x025BD, "bigtriangledown", "mathord", "big down triangle, open")
addSymbol(0x025BE, "blacktriangledown", "mathord", "down triangle, filled")
addSymbol(0x025BF, "triangledown", "mathord", "down triangle, open")
addSymbol(0x025C0, "blacktriangleleft", "mathord", "(large) left triangle, filled")
addSymbol(0x025C1, "triangleleft", "mathbin", "(large) left triangle, open; z notation domain restriction")
addSymbol(0x025C2, "smallblacktriangleleft", "mathord", "left triangle, filled")
addSymbol(0x025C3, "smalltriangleleft", "mathord", "left triangle, open")
addSymbol(0x025C4, "blackpointerleft", "mathord", "black left-pointing pointer")
addSymbol(0x025C5, "whitepointerleft", "mathord", "white left-pointing pointer")
addSymbol(0x025C6, "mdlgblkdiamond", "mathord", "black diamond")
addSymbol(0x025C7, "mdlgwhtdiamond", "mathord", "white diamond; diamond, open")
addSymbol(0x025C8, "blackinwhitediamond", "mathord", "white diamond containing black small diamond")
addSymbol(0x025C9, "fisheye", "mathord", "fisheye")
addSymbol(0x025CA, "mdlgwhtlozenge", "mathord", "lozenge or total mark")
addSymbol(0x025CB, "mdlgwhtcircle", "mathbin", "medium large circle")
addSymbol(0x025CC, "dottedcircle", "mathord", "dotted circle")
addSymbol(0x025CD, "circlevertfill", "mathord", "circle with vertical fill")
addSymbol(0x025CE, "bullseye", "mathord", "bullseye")
addSymbol(0x025CF, "mdlgblkcircle", "mathord", "circle, filled")
addSymbol(0x025D0, "circlelefthalfblack", "mathord", "circle, filled left half [harvey ball]")
addSymbol(0x025D1, "circlerighthalfblack", "mathord", "circle, filled right half")
addSymbol(0x025D2, "circlebottomhalfblack", "mathord", "circle, filled bottom half")
addSymbol(0x025D3, "circletophalfblack", "mathord", "circle, filled top half")
addSymbol(0x025D4, "circleurquadblack", "mathord", "circle with upper right quadrant black")
addSymbol(0x025D5, "blackcircleulquadwhite", "mathord", "circle with all but upper left quadrant black")
addSymbol(0x025D6, "blacklefthalfcircle", "mathord", "left half black circle")
addSymbol(0x025D7, "blackrighthalfcircle", "mathord", "right half black circle")
addSymbol(0x025D8, "inversebullet", "mathord", "inverse bullet ")
addSymbol(0x025D9, "inversewhitecircle", "mathord", "inverse white circle")
addSymbol(0x025DA, "invwhiteupperhalfcircle", "mathord", "upper half inverse white circle")
addSymbol(0x025DB, "invwhitelowerhalfcircle", "mathord", "lower half inverse white circle")
addSymbol(0x025DC, "ularc", "mathord", "upper left quadrant circular arc")
addSymbol(0x025DD, "urarc", "mathord", "upper right quadrant circular arc")
addSymbol(0x025DE, "lrarc", "mathord", "lower right quadrant circular arc")
addSymbol(0x025DF, "llarc", "mathord", "lower left quadrant circular arc")
addSymbol(0x025E0, "topsemicircle", "mathord", "upper half circle")
addSymbol(0x025E1, "botsemicircle", "mathord", "lower half circle")
addSymbol(0x025E2, "lrblacktriangle", "mathord", "lower right triangle, filled")
addSymbol(0x025E3, "llblacktriangle", "mathord", "lower left triangle, filled")
addSymbol(0x025E4, "ulblacktriangle", "mathord", "upper left triangle, filled")
addSymbol(0x025E5, "urblacktriangle", "mathord", "upper right triangle, filled")
addSymbol(0x025E6, "smwhtcircle", "mathord", "white bullet")
addSymbol(0x025E7, "squareleftblack", "mathord", "square, filled left half")
addSymbol(0x025E8, "squarerightblack", "mathord", "square, filled right half")
addSymbol(0x025E9, "squareulblack", "mathord", "square, filled top left corner")
addSymbol(0x025EA, "squarelrblack", "mathord", "square, filled bottom right corner")
addSymbol(0x025EB, "boxbar", "mathbin", "vertical bar in box")
addSymbol(0x025EC, "trianglecdot", "mathord", "triangle with centered dot")
addSymbol(0x025ED, "triangleleftblack", "mathord", "up-pointing triangle with left half black")
addSymbol(0x025EE, "trianglerightblack", "mathord", "up-pointing triangle with right half black")
addSymbol(0x025EF, "lgwhtcircle", "mathord", "large circle")
addSymbol(0x025F0, "squareulquad", "mathord", "white square with upper left quadrant")
addSymbol(0x025F1, "squarellquad", "mathord", "white square with lower left quadrant")
addSymbol(0x025F2, "squarelrquad", "mathord", "white square with lower right quadrant")
addSymbol(0x025F3, "squareurquad", "mathord", "white square with upper right quadrant")
addSymbol(0x025F4, "circleulquad", "mathord", "white circle with upper left quadrant")
addSymbol(0x025F5, "circlellquad", "mathord", "white circle with lower left quadrant")
addSymbol(0x025F6, "circlelrquad", "mathord", "white circle with lower right quadrant")
addSymbol(0x025F7, "circleurquad", "mathord", "white circle with upper right quadrant")
addSymbol(0x025F8, "ultriangle", "mathord", "upper left triangle")
addSymbol(0x025F9, "urtriangle", "mathord", "upper right triangle")
addSymbol(0x025FA, "lltriangle", "mathord", "lower left triangle")
addSymbol(0x025FB, "mdwhtsquare", "mathord", "white medium square")
addSymbol(0x025FC, "mdblksquare", "mathord", "black medium square")
addSymbol(0x025FD, "mdsmwhtsquare", "mathord", "white medium small square")
addSymbol(0x025FE, "mdsmblksquare", "mathord", "black medium small square")
addSymbol(0x025FF, "lrtriangle", "mathord", "lower right triangle")
addSymbol(0x02605, "bigstar", "mathord", "star, filled")
addSymbol(0x02606, "bigwhitestar", "mathord", "star, open")
addSymbol(0x02609, "astrosun", "mathord", "sun")
addSymbol(0x02621, "danger", "mathord", "dangerous bend (caution sign)")
addSymbol(0x0263B, "blacksmiley", "mathord", "black smiling face")
addSymbol(0x0263C, "sun", "mathord", "white sun with rays")
addSymbol(0x0263D, "rightmoon", "mathord", "first quarter moon")
addSymbol(0x0263E, "leftmoon", "mathord", "last quarter moon")
addSymbol(0x02640, "female", "mathord", "venus, female")
addSymbol(0x02642, "male", "mathord", "mars, male")
addSymbol(0x02660, "spadesuit", "mathord", "spades suit symbol")
addSymbol(0x02661, "heartsuit", "mathord", "heart suit symbol")
addSymbol(0x02662, "diamondsuit", "mathord", "diamond suit symbol")
addSymbol(0x02663, "clubsuit", "mathord", "club suit symbol")
addSymbol(0x02664, "varspadesuit", "mathord", "spade, white (card suit)")
addSymbol(0x02665, "varheartsuit", "mathord", "filled heart (card suit)")
addSymbol(0x02666, "vardiamondsuit", "mathord", "filled diamond (card suit)")
addSymbol(0x02667, "varclubsuit", "mathord", "club, white (card suit)")
addSymbol(0x02669, "quarternote", "mathord", "music note (sung text sign)")
addSymbol(0x0266A, "eighthnote", "mathord", "eighth note")
addSymbol(0x0266B, "twonotes", "mathord", "beamed eighth notes")
addSymbol(0x0266D, "flat", "mathord", "musical flat")
addSymbol(0x0266E, "natural", "mathord", "music natural")
addSymbol(0x0266F, "sharp", "mathord", "musical sharp")
addSymbol(0x0267E, "acidfree", "mathord", "permanent paper sign")
addSymbol(0x02680, "dicei", "mathord", "die face-1")
addSymbol(0x02681, "diceii", "mathord", "die face-2")
addSymbol(0x02682, "diceiii", "mathord", "die face-3")
addSymbol(0x02683, "diceiv", "mathord", "die face-4")
addSymbol(0x02684, "dicev", "mathord", "die face-5")
addSymbol(0x02685, "dicevi", "mathord", "die face-6")
addSymbol(0x02686, "circledrightdot", "mathord", "white circle with dot right")
addSymbol(0x02687, "circledtwodots", "mathord", "white circle with two dots")
addSymbol(0x02688, "blackcircledrightdot", "mathord", "black circle with white dot right")
addSymbol(0x02689, "blackcircledtwodots", "mathord", "black circle with two white dots")
addSymbol(0x026A5, "Hermaphrodite", "mathord", "male and female sign")
addSymbol(0x026AA, "mdwhtcircle", "mathord", "medium white circle")
addSymbol(0x026AB, "mdblkcircle", "mathord", "medium black circle")
addSymbol(0x026AC, "mdsmwhtcircle", "mathord", "medium small white circle")
addSymbol(0x026B2, "neuter", "mathord", "neuter")
addSymbol(0x02713, "checkmark", "mathord", "tick, check mark")
addSymbol(0x02720, "maltese", "mathord", "maltese cross")
addSymbol(0x0272A, "circledstar", "mathord", "circled white star")
addSymbol(0x02736, "varstar", "mathord", "six pointed black star")
addSymbol(0x0273D, "dingasterisk", "mathord", "heavy teardrop-spoked asterisk")
addSymbol(0x02772, "lbrbrak", "mathopen", "light left tortoise shell bracket ornament")
addSymbol(0x02773, "rbrbrak", "mathclose", "light right tortoise shell bracket ornament")
addSymbol(0x0279B, "draftingarrow", "mathord", "right arrow with bold head (drafting)")
addSymbol(0x027C0, "threedangle", "mathord", "three dimensional angle")
addSymbol(0x027C1, "whiteinwhitetriangle", "mathord", "white triangle containing small white triangle")
addSymbol(0x027C2, "perp", "mathrel", "perpendicular")
addSymbol(0x027C3, "subsetcirc", "mathord", "open subset")
addSymbol(0x027C4, "supsetcirc", "mathord", "open superset")
addSymbol(0x027C5, "lbag", "mathopen", "left s-shaped bag delimiter")
addSymbol(0x027C6, "rbag", "mathclose", "right s-shaped bag delimiter")
addSymbol(0x027C7, "veedot", "mathbin", "or with dot inside")
addSymbol(0x027C8, "bsolhsub", "mathrel", "reverse solidus preceding subset")
addSymbol(0x027C9, "suphsol", "mathrel", "superset preceding solidus")
addSymbol(0x027CC, "longdivision", "mathopen", "long division")
addSymbol(0x027D0, "diamondcdot", "mathord", "white diamond with centered dot")
addSymbol(0x027D1, "wedgedot", "mathbin", "and with dot")
addSymbol(0x027D2, "upin", "mathrel", "element of opening upwards")
addSymbol(0x027D3, "pullback", "mathrel", "lower right corner with dot")
addSymbol(0x027D4, "pushout", "mathrel", "upper left corner with dot")
addSymbol(0x027D5, "leftouterjoin", "mathop", "left outer join")
addSymbol(0x027D6, "rightouterjoin", "mathop", "right outer join")
addSymbol(0x027D7, "fullouterjoin", "mathop", "full outer join")
addSymbol(0x027D8, "bigbot", "mathop", "large up tack")
addSymbol(0x027D9, "bigtop", "mathop", "large down tack")
addSymbol(0x027DA, "DashVDash", "mathrel", "left and right double turnstile")
addSymbol(0x027DB, "dashVdash", "mathrel", "left and right tack")
addSymbol(0x027DC, "multimapinv", "mathrel", "left multimap")
addSymbol(0x027DD, "vlongdash", "mathrel", "long left tack")
addSymbol(0x027DE, "longdashv", "mathrel", "long right tack")
addSymbol(0x027DF, "cirbot", "mathrel", "up tack with circle above")
addSymbol(0x027E0, "lozengeminus", "mathbin", "lozenge divided by horizontal rule")
addSymbol(0x027E1, "concavediamond", "mathbin", "white concave-sided diamond")
addSymbol(0x027E2, "concavediamondtickleft", "mathbin", "white concave-sided diamond with leftwards tick")
addSymbol(0x027E3, "concavediamondtickright", "mathbin", "white concave-sided diamond with rightwards tick")
addSymbol(0x027E4, "whitesquaretickleft", "mathbin", "white square with leftwards tick")
addSymbol(0x027E5, "whitesquaretickright", "mathbin", "white square with rightwards tick")
addSymbol(0x027E6, "lBrack", "mathopen", "mathematical left white square bracket")
addSymbol(0x027E7, "rBrack", "mathclose", "mathematical right white square bracket")
addSymbol(0x027E8, "langle", "mathopen", "mathematical left angle bracket")
addSymbol(0x027E9, "rangle", "mathclose", "mathematical right angle bracket")
addSymbol(0x027EA, "lAngle", "mathopen", "mathematical left double angle bracket")
addSymbol(0x027EB, "rAngle", "mathclose", "mathematical right double angle bracket")
addSymbol(0x027EC, "Lbrbrak", "mathopen", "mathematical left white tortoise shell bracket")
addSymbol(0x027ED, "Rbrbrak", "mathclose", "mathematical right white tortoise shell bracket")
addSymbol(0x027EE, "lgroup", "mathopen", "mathematical left flattened parenthesis")
addSymbol(0x027EF, "rgroup", "mathclose", "mathematical right flattened parenthesis")
addSymbol(0x027F0, "UUparrow", "mathrel", "upwards quadruple arrow")
addSymbol(0x027F1, "DDownarrow", "mathrel", "downwards quadruple arrow")
addSymbol(0x027F2, "acwgapcirclearrow", "mathrel", "anticlockwise gapped circle arrow")
addSymbol(0x027F3, "cwgapcirclearrow", "mathrel", "clockwise gapped circle arrow")
addSymbol(0x027F4, "rightarrowonoplus", "mathrel", "right arrow with circled plus")
addSymbol(0x027F5, "longleftarrow", "mathrel", "long leftwards arrow")
addSymbol(0x027F6, "longrightarrow", "mathrel", "long rightwards arrow")
addSymbol(0x027F7, "longleftrightarrow", "mathrel", "long left right arrow")
addSymbol(0x027F8, "Longleftarrow", "mathrel", "long leftwards double arrow")
addSymbol(0x027F9, "Longrightarrow", "mathrel", "long rightwards double arrow")
addSymbol(0x027FA, "Longleftrightarrow", "mathrel", "long left right double arrow")
addSymbol(0x027FB, "longmapsfrom", "mathrel", "long leftwards arrow from bar")
addSymbol(0x027FC, "longmapsto", "mathrel", "long rightwards arrow from bar")
addSymbol(0x027FD, "Longmapsfrom", "mathrel", "long leftwards double arrow from bar")
addSymbol(0x027FE, "Longmapsto", "mathrel", "long rightwards double arrow from bar")
addSymbol(0x027FF, "longrightsquigarrow", "mathrel", "long rightwards squiggle arrow")
addSymbol(0x02900, "nvtwoheadrightarrow", "mathrel", "rightwards two-headed arrow with vertical stroke")
addSymbol(0x02901, "nVtwoheadrightarrow", "mathrel", "rightwards two-headed arrow with double vertical stroke")
addSymbol(0x02902, "nvLeftarrow", "mathrel", "leftwards double arrow with vertical stroke")
addSymbol(0x02903, "nvRightarrow", "mathrel", "rightwards double arrow with vertical stroke")
addSymbol(0x02904, "nvLeftrightarrow", "mathrel", "left right double arrow with vertical stroke")
addSymbol(0x02905, "twoheadmapsto", "mathrel", "rightwards two-headed arrow from bar")
addSymbol(0x02906, "Mapsfrom", "mathrel", "leftwards double arrow from bar")
addSymbol(0x02907, "Mapsto", "mathrel", "rightwards double arrow from bar")
addSymbol(0x02908, "downarrowbarred", "mathrel", "downwards arrow with horizontal stroke")
addSymbol(0x02909, "uparrowbarred", "mathrel", "upwards arrow with horizontal stroke")
addSymbol(0x0290A, "Uuparrow", "mathrel", "upwards triple arrow")
addSymbol(0x0290B, "Ddownarrow", "mathrel", "downwards triple arrow")
addSymbol(0x0290C, "leftbkarrow", "mathrel", "leftwards double dash arrow")
addSymbol(0x0290D, "rightbkarrow", "mathrel", "rightwards double dash arrow")
addSymbol(0x0290E, "leftdbkarrow", "mathrel", "leftwards triple dash arrow")
addSymbol(0x0290F, "dbkarrow", "mathrel", "rightwards triple dash arrow")
addSymbol(0x02910, "drbkarrow", "mathrel", "rightwards two-headed triple dash arrow")
addSymbol(0x02911, "rightdotarrow", "mathrel", "rightwards arrow with dotted stem")
addSymbol(0x02912, "baruparrow", "mathrel", "upwards arrow to bar")
addSymbol(0x02913, "downarrowbar", "mathrel", "downwards arrow to bar")
addSymbol(0x02914, "nvrightarrowtail", "mathrel", "rightwards arrow with tail with vertical stroke")
addSymbol(0x02915, "nVrightarrowtail", "mathrel", "rightwards arrow with tail with double vertical stroke")
addSymbol(0x02916, "twoheadrightarrowtail", "mathrel", "rightwards two-headed arrow with tail")
addSymbol(0x02917, "nvtwoheadrightarrowtail", "mathrel", "rightwards two-headed arrow with tail with vertical stroke")
addSymbol(
   0x02918,
   "nVtwoheadrightarrowtail",
   "mathrel",
   "rightwards two-headed arrow with tail with double vertical stroke"
)
addSymbol(0x02919, "lefttail", "mathrel", "leftwards arrow-tail")
addSymbol(0x0291A, "righttail", "mathrel", "rightwards arrow-tail")
addSymbol(0x0291B, "leftdbltail", "mathrel", "leftwards double arrow-tail")
addSymbol(0x0291C, "rightdbltail", "mathrel", "rightwards double arrow-tail")
addSymbol(0x0291D, "diamondleftarrow", "mathrel", "leftwards arrow to black diamond")
addSymbol(0x0291E, "rightarrowdiamond", "mathrel", "rightwards arrow to black diamond")
addSymbol(0x0291F, "diamondleftarrowbar", "mathrel", "leftwards arrow from bar to black diamond")
addSymbol(0x02920, "barrightarrowdiamond", "mathrel", "rightwards arrow from bar to black diamond")
addSymbol(0x02921, "nwsearrow", "mathrel", "north west and south east arrow")
addSymbol(0x02922, "neswarrow", "mathrel", "north east and south west arrow")
addSymbol(0x02923, "hknwarrow", "mathrel", "north west arrow with hook")
addSymbol(0x02924, "hknearrow", "mathrel", "north east arrow with hook")
addSymbol(0x02925, "hksearrow", "mathrel", "south east arrow with hook")
addSymbol(0x02926, "hkswarrow", "mathrel", "south west arrow with hook")
addSymbol(0x02927, "tona", "mathrel", "north west arrow and north east arrow")
addSymbol(0x02928, "toea", "mathrel", "north east arrow and south east arrow")
addSymbol(0x02929, "tosa", "mathrel", "south east arrow and south west arrow")
addSymbol(0x0292A, "towa", "mathrel", "south west arrow and north west arrow")
addSymbol(0x0292B, "rdiagovfdiag", "mathord", "rising diagonal crossing falling diagonal")
addSymbol(0x0292C, "fdiagovrdiag", "mathord", "falling diagonal crossing rising diagonal")
addSymbol(0x0292D, "seovnearrow", "mathord", "south east arrow crossing north east arrow")
addSymbol(0x0292E, "neovsearrow", "mathord", "north east arrow crossing south east arrow")
addSymbol(0x0292F, "fdiagovnearrow", "mathord", "falling diagonal crossing north east arrow")
addSymbol(0x02930, "rdiagovsearrow", "mathord", "rising diagonal crossing south east arrow")
addSymbol(0x02931, "neovnwarrow", "mathord", "north east arrow crossing north west arrow")
addSymbol(0x02932, "nwovnearrow", "mathord", "north west arrow crossing north east arrow")
addSymbol(0x02933, "rightcurvedarrow", "mathrel", "wave arrow pointing directly right")
addSymbol(0x02934, "uprightcurvearrow", "mathord", "arrow pointing rightwards then curving upwards")
addSymbol(0x02935, "downrightcurvedarrow", "mathord", "arrow pointing rightwards then curving downwards")
addSymbol(0x02936, "leftdowncurvedarrow", "mathrel", "arrow pointing downwards then curving leftwards")
addSymbol(0x02937, "rightdowncurvedarrow", "mathrel", "arrow pointing downwards then curving rightwards")
addSymbol(0x02938, "cwrightarcarrow", "mathrel", "right-side arc clockwise arrow")
addSymbol(0x02939, "acwleftarcarrow", "mathrel", "left-side arc anticlockwise arrow")
addSymbol(0x0293A, "acwoverarcarrow", "mathrel", "top arc anticlockwise arrow")
addSymbol(0x0293B, "acwunderarcarrow", "mathrel", "bottom arc anticlockwise arrow")
addSymbol(0x0293C, "curvearrowrightminus", "mathrel", "top arc clockwise arrow with minus")
addSymbol(0x0293D, "curvearrowleftplus", "mathrel", "top arc anticlockwise arrow with plus")
addSymbol(0x0293E, "cwundercurvearrow", "mathrel", "lower right semicircular clockwise arrow")
addSymbol(0x0293F, "ccwundercurvearrow", "mathrel", "lower left semicircular anticlockwise arrow")
addSymbol(0x02940, "acwcirclearrow", "mathrel", "anticlockwise closed circle arrow")
addSymbol(0x02941, "cwcirclearrow", "mathrel", "clockwise closed circle arrow")
addSymbol(0x02942, "rightarrowshortleftarrow", "mathrel", "rightwards arrow above short leftwards arrow")
addSymbol(0x02943, "leftarrowshortrightarrow", "mathrel", "leftwards arrow above short rightwards arrow")
addSymbol(0x02944, "shortrightarrowleftarrow", "mathrel", "short rightwards arrow above leftwards arrow")
addSymbol(0x02945, "rightarrowplus", "mathrel", "rightwards arrow with plus below")
addSymbol(0x02946, "leftarrowplus", "mathrel", "leftwards arrow with plus below")
addSymbol(0x02947, "rightarrowx", "mathrel", "rightwards arrow through x")
addSymbol(0x02948, "leftrightarrowcircle", "mathrel", "left right arrow through small circle")
addSymbol(0x02949, "twoheaduparrowcircle", "mathrel", "upwards two-headed arrow from small circle")
addSymbol(0x0294A, "leftrightharpoonupdown", "mathrel", "left barb up right barb down harpoon")
addSymbol(0x0294B, "leftrightharpoondownup", "mathrel", "left barb down right barb up harpoon")
addSymbol(0x0294C, "updownharpoonrightleft", "mathrel", "up barb right down barb left harpoon")
addSymbol(0x0294D, "updownharpoonleftright", "mathrel", "up barb left down barb right harpoon")
addSymbol(0x0294E, "leftrightharpoonupup", "mathrel", "left barb up right barb up harpoon")
addSymbol(0x0294F, "updownharpoonrightright", "mathrel", "up barb right down barb right harpoon")
addSymbol(0x02950, "leftrightharpoondowndown", "mathrel", "left barb down right barb down harpoon")
addSymbol(0x02951, "updownharpoonleftleft", "mathrel", "up barb left down barb left harpoon")
addSymbol(0x02952, "barleftharpoonup", "mathrel", "leftwards harpoon with barb up to bar")
addSymbol(0x02953, "rightharpoonupbar", "mathrel", "rightwards harpoon with barb up to bar")
addSymbol(0x02954, "barupharpoonright", "mathrel", "upwards harpoon with barb right to bar")
addSymbol(0x02955, "downharpoonrightbar", "mathrel", "downwards harpoon with barb right to bar")
addSymbol(0x02956, "barleftharpoondown", "mathrel", "leftwards harpoon with barb down to bar")
addSymbol(0x02957, "rightharpoondownbar", "mathrel", "rightwards harpoon with barb down to bar")
addSymbol(0x02958, "barupharpoonleft", "mathrel", "upwards harpoon with barb left to bar")
addSymbol(0x02959, "downharpoonleftbar", "mathrel", "downwards harpoon with barb left to bar")
addSymbol(0x0295A, "leftharpoonupbar", "mathrel", "leftwards harpoon with barb up from bar")
addSymbol(0x0295B, "barrightharpoonup", "mathrel", "rightwards harpoon with barb up from bar")
addSymbol(0x0295C, "upharpoonrightbar", "mathrel", "upwards harpoon with barb right from bar")
addSymbol(0x0295D, "bardownharpoonright", "mathrel", "downwards harpoon with barb right from bar")
addSymbol(0x0295E, "leftharpoondownbar", "mathrel", "leftwards harpoon with barb down from bar")
addSymbol(0x0295F, "barrightharpoondown", "mathrel", "rightwards harpoon with barb down from bar")
addSymbol(0x02960, "upharpoonleftbar", "mathrel", "upwards harpoon with barb left from bar")
addSymbol(0x02961, "bardownharpoonleft", "mathrel", "downwards harpoon with barb left from bar")
addSymbol(
   0x02962,
   "leftharpoonsupdown",
   "mathrel",
   "leftwards harpoon with barb up above leftwards harpoon with barb down"
)
addSymbol(
   0x02963,
   "upharpoonsleftright",
   "mathrel",
   "upwards harpoon with barb left beside upwards harpoon with barb right"
)
addSymbol(
   0x02964,
   "rightharpoonsupdown",
   "mathrel",
   "rightwards harpoon with barb up above rightwards harpoon with barb down"
)
addSymbol(
   0x02965,
   "downharpoonsleftright",
   "mathrel",
   "downwards harpoon with barb left beside downwards harpoon with barb right"
)
addSymbol(
   0x02966,
   "leftrightharpoonsup",
   "mathrel",
   "leftwards harpoon with barb up above rightwards harpoon with barb up"
)
addSymbol(
   0x02967,
   "leftrightharpoonsdown",
   "mathrel",
   "leftwards harpoon with barb down above rightwards harpoon with barb down"
)
addSymbol(
   0x02968,
   "rightleftharpoonsup",
   "mathrel",
   "rightwards harpoon with barb up above leftwards harpoon with barb up"
)
addSymbol(
   0x02969,
   "rightleftharpoonsdown",
   "mathrel",
   "rightwards harpoon with barb down above leftwards harpoon with barb down"
)
addSymbol(0x0296A, "leftharpoonupdash", "mathrel", "leftwards harpoon with barb up above long dash")
addSymbol(0x0296B, "dashleftharpoondown", "mathrel", "leftwards harpoon with barb down below long dash")
addSymbol(0x0296C, "rightharpoonupdash", "mathrel", "rightwards harpoon with barb up above long dash")
addSymbol(0x0296D, "dashrightharpoondown", "mathrel", "rightwards harpoon with barb down below long dash")
addSymbol(
   0x0296E,
   "updownharpoonsleftright",
   "mathrel",
   "upwards harpoon with barb left beside downwards harpoon with barb right"
)
addSymbol(
   0x0296F,
   "downupharpoonsleftright",
   "mathrel",
   "downwards harpoon with barb left beside upwards harpoon with barb right"
)
addSymbol(0x02970, "rightimply", "mathrel", "right double arrow with rounded head")
addSymbol(0x02971, "equalrightarrow", "mathrel", "equals sign above rightwards arrow")
addSymbol(0x02972, "similarrightarrow", "mathrel", "tilde operator above rightwards arrow")
addSymbol(0x02973, "leftarrowsimilar", "mathrel", "leftwards arrow above tilde operator")
addSymbol(0x02974, "rightarrowsimilar", "mathrel", "rightwards arrow above tilde operator")
addSymbol(0x02975, "rightarrowapprox", "mathrel", "rightwards arrow above almost equal to")
addSymbol(0x02976, "ltlarr", "mathrel", "less-than above leftwards arrow")
addSymbol(0x02977, "leftarrowless", "mathrel", "leftwards arrow through less-than")
addSymbol(0x02978, "gtrarr", "mathrel", "greater-than above rightwards arrow")
addSymbol(0x02979, "subrarr", "mathrel", "subset above rightwards arrow")
addSymbol(0x0297A, "leftarrowsubset", "mathrel", "leftwards arrow through subset")
addSymbol(0x0297B, "suplarr", "mathrel", "superset above leftwards arrow")
addSymbol(0x0297C, "leftfishtail", "mathrel", "left fish tail")
addSymbol(0x0297D, "rightfishtail", "mathrel", "right fish tail")
addSymbol(0x0297E, "upfishtail", "mathrel", "up fish tail")
addSymbol(0x0297F, "downfishtail", "mathrel", "down fish tail")
addSymbol(0x02980, "Vvert", "mathfence", "triple vertical bar delimiter")
addSymbol(0x02981, "mdsmblkcircle", "mathord", "z notation spot")
addSymbol(0x02982, "typecolon", "mathrel", "z notation type colon")
addSymbol(0x02983, "lBrace", "mathopen", "left white curly bracket")
addSymbol(0x02984, "rBrace", "mathclose", "right white curly bracket")
addSymbol(0x02985, "lParen", "mathopen", "left white parenthesis")
addSymbol(0x02986, "rParen", "mathclose", "right white parenthesis")
addSymbol(0x02987, "llparenthesis", "mathopen", "z notation left image bracket")
addSymbol(0x02988, "rrparenthesis", "mathclose", "z notation right image bracket")
addSymbol(0x02989, "llangle", "mathopen", "z notation left binding bracket")
addSymbol(0x0298A, "rrangle", "mathclose", "z notation right binding bracket")
addSymbol(0x0298B, "lbrackubar", "mathopen", "left square bracket with underbar")
addSymbol(0x0298C, "rbrackubar", "mathclose", "right square bracket with underbar")
addSymbol(0x0298D, "lbrackultick", "mathopen", "left square bracket with tick in top corner")
addSymbol(0x0298E, "rbracklrtick", "mathclose", "right square bracket with tick in bottom corner")
addSymbol(0x0298F, "lbracklltick", "mathopen", "left square bracket with tick in bottom corner")
addSymbol(0x02990, "rbrackurtick", "mathclose", "right square bracket with tick in top corner")
addSymbol(0x02991, "langledot", "mathopen", "left angle bracket with dot")
addSymbol(0x02992, "rangledot", "mathclose", "right angle bracket with dot")
addSymbol(0x02993, "lparenless", "mathopen", "left arc less-than bracket")
addSymbol(0x02994, "rparengtr", "mathclose", "right arc greater-than bracket")
addSymbol(0x02995, "Lparengtr", "mathopen", "double left arc greater-than bracket")
addSymbol(0x02996, "Rparenless", "mathclose", "double right arc less-than bracket")
addSymbol(0x02997, "lblkbrbrak", "mathopen", "left black tortoise shell bracket")
addSymbol(0x02998, "rblkbrbrak", "mathclose", "right black tortoise shell bracket")
addSymbol(0x02999, "fourvdots", "mathord", "dotted fence")
addSymbol(0x0299A, "vzigzag", "mathord", "vertical zigzag line")
addSymbol(0x0299B, "measuredangleleft", "mathord", "measured angle opening left")
addSymbol(0x0299C, "rightanglesqr", "mathord", "right angle variant with square")
addSymbol(0x0299D, "rightanglemdot", "mathord", "measured right angle with dot")
addSymbol(0x0299E, "angles", "mathord", "angle with s inside")
addSymbol(0x0299F, "angdnr", "mathord", "acute angle")
addSymbol(0x029A0, "gtlpar", "mathord", "spherical angle opening left")
addSymbol(0x029A1, "sphericalangleup", "mathord", "spherical angle opening up")
addSymbol(0x029A2, "turnangle", "mathord", "turned angle")
addSymbol(0x029A3, "revangle", "mathord", "reversed angle")
addSymbol(0x029A4, "angleubar", "mathord", "angle with underbar")
addSymbol(0x029A5, "revangleubar", "mathord", "reversed angle with underbar")
addSymbol(0x029A6, "wideangledown", "mathord", "oblique angle opening up")
addSymbol(0x029A7, "wideangleup", "mathord", "oblique angle opening down")
addSymbol(0x029A8, "measanglerutone", "mathord", "measured angle with open arm ending in arrow pointing up and right")
addSymbol(0x029A9, "measanglelutonw", "mathord", "measured angle with open arm ending in arrow pointing up and left")
addSymbol(0x029AA, "measanglerdtose", "mathord", "measured angle with open arm ending in arrow pointing down and right")
addSymbol(0x029AB, "measangleldtosw", "mathord", "measured angle with open arm ending in arrow pointing down and left")
addSymbol(0x029AC, "measangleurtone", "mathord", "measured angle with open arm ending in arrow pointing right and up")
addSymbol(0x029AD, "measangleultonw", "mathord", "measured angle with open arm ending in arrow pointing left and up")
addSymbol(0x029AE, "measangledrtose", "mathord", "measured angle with open arm ending in arrow pointing right and down")
addSymbol(0x029AF, "measangledltosw", "mathord", "measured angle with open arm ending in arrow pointing left and down")
addSymbol(0x029B0, "revemptyset", "mathord", "reversed empty set")
addSymbol(0x029B1, "emptysetobar", "mathord", "empty set with overbar")
addSymbol(0x029B2, "emptysetocirc", "mathord", "empty set with small circle above")
addSymbol(0x029B3, "emptysetoarr", "mathord", "empty set with right arrow above")
addSymbol(0x029B4, "emptysetoarrl", "mathord", "empty set with left arrow above")
addSymbol(0x029B5, "circlehbar", "mathbin", "circle with horizontal bar")
addSymbol(0x029B6, "circledvert", "mathbin", "circled vertical bar")
addSymbol(0x029B7, "circledparallel", "mathbin", "circled parallel")
addSymbol(0x029B8, "obslash", "mathbin", "circled reverse solidus")
addSymbol(0x029B9, "operp", "mathbin", "circled perpendicular")
addSymbol(0x029BA, "obot", "mathord", "circle divided by horizontal bar and top half divided by vertical bar")
addSymbol(0x029BB, "olcross", "mathord", "circle with superimposed x")
addSymbol(0x029BC, "odotslashdot", "mathord", "circled anticlockwise-rotated division sign")
addSymbol(0x029BD, "uparrowoncircle", "mathord", "up arrow through circle")
addSymbol(0x029BE, "circledwhitebullet", "mathord", "circled white bullet")
addSymbol(0x029BF, "circledbullet", "mathord", "circled bullet")
addSymbol(0x029C0, "olessthan", "mathbin", "circled less-than")
addSymbol(0x029C1, "ogreaterthan", "mathbin", "circled greater-than")
addSymbol(0x029C2, "cirscir", "mathord", "circle with small circle to the right")
addSymbol(0x029C3, "cirE", "mathord", "circle with two horizontal strokes to the right")
addSymbol(0x029C4, "boxdiag", "mathbin", "squared rising diagonal slash")
addSymbol(0x029C5, "boxbslash", "mathbin", "squared falling diagonal slash")
addSymbol(0x029C6, "boxast", "mathbin", "squared asterisk")
addSymbol(0x029C7, "boxcircle", "mathbin", "squared small circle")
addSymbol(0x029C8, "boxbox", "mathbin", "squared square")
addSymbol(0x029C9, "boxonbox", "mathord", "two joined squares")
addSymbol(0x029CA, "triangleodot", "mathord", "triangle with dot above")
addSymbol(0x029CB, "triangleubar", "mathord", "triangle with underbar")
addSymbol(0x029CC, "triangles", "mathord", "s in triangle")
addSymbol(0x029CD, "triangleserifs", "mathbin", "triangle with serifs at bottom")
addSymbol(0x029CE, "rtriltri", "mathrel", "right triangle above left triangle")
addSymbol(0x029CF, "ltrivb", "mathrel", "left triangle beside vertical bar")
addSymbol(0x029D0, "vbrtri", "mathrel", "vertical bar beside right triangle")
addSymbol(0x029D1, "lfbowtie", "mathrel", "left black bowtie")
addSymbol(0x029D2, "rfbowtie", "mathrel", "right black bowtie")
addSymbol(0x029D3, "fbowtie", "mathrel", "black bowtie")
addSymbol(0x029D4, "lftimes", "mathrel", "left black times")
addSymbol(0x029D5, "rftimes", "mathrel", "right black times")
addSymbol(0x029D6, "hourglass", "mathbin", "white hourglass")
addSymbol(0x029D7, "blackhourglass", "mathbin", "black hourglass")
addSymbol(0x029D8, "lvzigzag", "mathopen", "left wiggly fence")
addSymbol(0x029D9, "rvzigzag", "mathclose", "right wiggly fence")
addSymbol(0x029DA, "Lvzigzag", "mathopen", "left double wiggly fence")
addSymbol(0x029DB, "Rvzigzag", "mathclose", "right double wiggly fence")
addSymbol(0x029DC, "iinfin", "mathord", "incomplete infinity")
addSymbol(0x029DD, "tieinfty", "mathord", "tie over infinity")
addSymbol(0x029DE, "nvinfty", "mathord", "infinity negated with vertical bar")
addSymbol(0x029DF, "dualmap", "mathrel", "double-ended multimap")
addSymbol(0x029E0, "laplac", "mathord", "square with contoured outline")
addSymbol(0x029E1, "lrtriangleeq", "mathrel", "increases as")
addSymbol(0x029E2, "shuffle", "mathbin", "shuffle product")
addSymbol(0x029E3, "eparsl", "mathrel", "equals sign and slanted parallel")
addSymbol(0x029E4, "smeparsl", "mathrel", "equals sign and slanted parallel with tilde above")
addSymbol(0x029E5, "eqvparsl", "mathrel", "identical to and slanted parallel")
addSymbol(0x029E6, "gleichstark", "mathrel", "gleich stark")
addSymbol(0x029E7, "thermod", "mathord", "thermodynamic")
addSymbol(0x029E8, "downtriangleleftblack", "mathord", "down-pointing triangle with left half black")
addSymbol(0x029E9, "downtrianglerightblack", "mathord", "down-pointing triangle with right half black")
addSymbol(0x029EA, "blackdiamonddownarrow", "mathord", "black diamond with down arrow")
addSymbol(0x029EB, "mdlgblklozenge", "mathbin", "black lozenge")
addSymbol(0x029EC, "circledownarrow", "mathord", "white circle with down arrow")
addSymbol(0x029ED, "blackcircledownarrow", "mathord", "black circle with down arrow")
addSymbol(0x029EE, "errbarsquare", "mathord", "error-barred white square")
addSymbol(0x029EF, "errbarblacksquare", "mathord", "error-barred black square")
addSymbol(0x029F0, "errbardiamond", "mathord", "error-barred white diamond")
addSymbol(0x029F1, "errbarblackdiamond", "mathord", "error-barred black diamond")
addSymbol(0x029F2, "errbarcircle", "mathord", "error-barred white circle")
addSymbol(0x029F3, "errbarblackcircle", "mathord", "error-barred black circle")
addSymbol(0x029F4, "ruledelayed", "mathrel", "rule-delayed")
addSymbol(0x029F5, "setminus", "mathbin", "reverse solidus operator")
addSymbol(0x029F6, "dsol", "mathbin", "solidus with overbar")
addSymbol(0x029F7, "rsolbar", "mathbin", "reverse solidus with horizontal stroke")
addSymbol(0x029F8, "xsol", "mathop", "big solidus")
addSymbol(0x029F9, "xbsol", "mathop", "big reverse solidus")
addSymbol(0x029FA, "doubleplus", "mathbin", "double plus")
addSymbol(0x029FB, "tripleplus", "mathbin", "triple plus")
addSymbol(0x029FC, "lcurvyangle", "mathopen", "left pointing curved angle bracket")
addSymbol(0x029FD, "rcurvyangle", "mathclose", "right pointing curved angle bracket")
addSymbol(0x029FE, "tplus", "mathbin", "tiny")
addSymbol(0x029FF, "tminus", "mathbin", "miny")
addSymbol(0x02A00, "bigodot", "mathop", "n-ary circled dot operator")
addSymbol(0x02A01, "bigoplus", "mathop", "n-ary circled plus operator")
addSymbol(0x02A02, "bigotimes", "mathop", "n-ary circled times operator")
addSymbol(0x02A03, "bigcupdot", "mathop", "n-ary union operator with dot")
addSymbol(0x02A04, "biguplus", "mathop", "n-ary union operator with plus")
addSymbol(0x02A05, "bigsqcap", "mathop", "n-ary square intersection operator")
addSymbol(0x02A06, "bigsqcup", "mathop", "n-ary square union operator")
addSymbol(0x02A07, "conjquant", "mathop", "two logical and operator")
addSymbol(0x02A08, "disjquant", "mathop", "two logical or operator")
addSymbol(0x02A09, "bigtimes", "mathop", "n-ary times operator")
addSymbol(0x02A0A, "modtwosum", "mathop", "modulo two sum")
addSymbol(0x02A0B, "sumint", "mathop", "summation with integral")
addSymbol(0x02A0C, "iiiint", "mathop", "quadruple integral operator")
addSymbol(0x02A0D, "intbar", "mathop", "finite part integral")
addSymbol(0x02A0E, "intBar", "mathop", "integral with double stroke")
addSymbol(0x02A0F, "fint", "mathop", "integral average with slash")
addSymbol(0x02A10, "cirfnint", "mathop", "circulation function")
addSymbol(0x02A11, "awint", "mathop", "anticlockwise integration")
addSymbol(0x02A12, "rppolint", "mathop", "line integration with rectangular path around pole")
addSymbol(0x02A13, "scpolint", "mathop", "line integration with semicircular path around pole")
addSymbol(0x02A14, "npolint", "mathop", "line integration not including the pole")
addSymbol(0x02A15, "pointint", "mathop", "integral around a point operator")
addSymbol(0x02A16, "sqint", "mathop", "quaternion integral operator")
addSymbol(0x02A17, "intlarhk", "mathop", "integral with leftwards arrow with hook")
addSymbol(0x02A18, "intx", "mathop", "integral with times sign")
addSymbol(0x02A19, "intcap", "mathop", "integral with intersection")
addSymbol(0x02A1A, "intcup", "mathop", "integral with union")
addSymbol(0x02A1B, "upint", "mathop", "integral with overbar")
addSymbol(0x02A1C, "lowint", "mathop", "integral with underbar")
addSymbol(0x02A1D, "Join", "mathop", "join")
addSymbol(0x02A1E, "bigtriangleleft", "mathop", "large left triangle operator")
addSymbol(0x02A1F, "zcmp", "mathop", "z notation schema composition")
addSymbol(0x02A20, "zpipe", "mathop", "z notation schema piping")
addSymbol(0x02A21, "zproject", "mathop", "z notation schema projection")
addSymbol(0x02A22, "ringplus", "mathbin", "plus sign with small circle above")
addSymbol(0x02A23, "plushat", "mathbin", "plus sign with circumflex accent above")
addSymbol(0x02A24, "simplus", "mathbin", "plus sign with tilde above")
addSymbol(0x02A25, "plusdot", "mathbin", "plus sign with dot below")
addSymbol(0x02A26, "plussim", "mathbin", "plus sign with tilde below")
addSymbol(0x02A27, "plussubtwo", "mathbin", "plus sign with subscript two")
addSymbol(0x02A28, "plustrif", "mathbin", "plus sign with black triangle")
addSymbol(0x02A29, "commaminus", "mathbin", "minus sign with comma above")
addSymbol(0x02A2A, "minusdot", "mathbin", "minus sign with dot below")
addSymbol(0x02A2B, "minusfdots", "mathbin", "minus sign with falling dots")
addSymbol(0x02A2C, "minusrdots", "mathbin", "minus sign with rising dots")
addSymbol(0x02A2D, "opluslhrim", "mathbin", "plus sign in left half circle")
addSymbol(0x02A2E, "oplusrhrim", "mathbin", "plus sign in right half circle")
addSymbol(0x02A2F, "vectimes", "mathbin", "vector or cross product")
addSymbol(0x02A30, "dottimes", "mathbin", "multiplication sign with dot above")
addSymbol(0x02A31, "timesbar", "mathbin", "multiplication sign with underbar")
addSymbol(0x02A32, "btimes", "mathbin", "semidirect product with bottom closed")
addSymbol(0x02A33, "smashtimes", "mathbin", "smash product")
addSymbol(0x02A34, "otimeslhrim", "mathbin", "multiplication sign in left half circle")
addSymbol(0x02A35, "otimesrhrim", "mathbin", "multiplication sign in right half circle")
addSymbol(0x02A36, "otimeshat", "mathbin", "circled multiplication sign with circumflex accent")
addSymbol(0x02A37, "Otimes", "mathbin", "multiplication sign in double circle")
addSymbol(0x02A38, "odiv", "mathbin", "circled division sign")
addSymbol(0x02A39, "triangleplus", "mathbin", "plus sign in triangle")
addSymbol(0x02A3A, "triangleminus", "mathbin", "minus sign in triangle")
addSymbol(0x02A3B, "triangletimes", "mathbin", "multiplication sign in triangle")
addSymbol(0x02A3C, "intprod", "mathbin", "interior product")
addSymbol(0x02A3D, "intprodr", "mathbin", "righthand interior product")
addSymbol(0x02A3E, "fcmp", "mathbin", "z notation relational composition")
addSymbol(0x02A3F, "amalg", "mathbin", "amalgamation or coproduct")
addSymbol(0x02A40, "capdot", "mathbin", "intersection with dot")
addSymbol(0x02A41, "uminus", "mathbin", "union with minus sign")
addSymbol(0x02A42, "barcup", "mathbin", "union with overbar")
addSymbol(0x02A43, "barcap", "mathbin", "intersection with overbar")
addSymbol(0x02A44, "capwedge", "mathbin", "intersection with logical and")
addSymbol(0x02A45, "cupvee", "mathbin", "union with logical or")
addSymbol(0x02A46, "cupovercap", "mathbin", "union above intersection")
addSymbol(0x02A47, "capovercup", "mathbin", "intersection above union")
addSymbol(0x02A48, "cupbarcap", "mathbin", "union above bar above intersection")
addSymbol(0x02A49, "capbarcup", "mathbin", "intersection above bar above union")
addSymbol(0x02A4A, "twocups", "mathbin", "union beside and joined with union")
addSymbol(0x02A4B, "twocaps", "mathbin", "intersection beside and joined with intersection")
addSymbol(0x02A4C, "closedvarcup", "mathbin", "closed union with serifs")
addSymbol(0x02A4D, "closedvarcap", "mathbin", "closed intersection with serifs")
addSymbol(0x02A4E, "Sqcap", "mathbin", "double square intersection")
addSymbol(0x02A4F, "Sqcup", "mathbin", "double square union")
addSymbol(0x02A50, "closedvarcupsmashprod", "mathbin", "closed union with serifs and smash product")
addSymbol(0x02A51, "wedgeodot", "mathbin", "logical and with dot above")
addSymbol(0x02A52, "veeodot", "mathbin", "logical or with dot above")
addSymbol(0x02A53, "Wedge", "mathbin", "double logical and")
addSymbol(0x02A54, "Vee", "mathbin", "double logical or")
addSymbol(0x02A55, "wedgeonwedge", "mathbin", "two intersecting logical and")
addSymbol(0x02A56, "veeonvee", "mathbin", "two intersecting logical or")
addSymbol(0x02A57, "bigslopedvee", "mathbin", "sloping large or")
addSymbol(0x02A58, "bigslopedwedge", "mathbin", "sloping large and")
addSymbol(0x02A59, "veeonwedge", "mathrel", "logical or overlapping logical and")
addSymbol(0x02A5A, "wedgemidvert", "mathbin", "logical and with middle stem")
addSymbol(0x02A5B, "veemidvert", "mathbin", "logical or with middle stem")
addSymbol(0x02A5C, "midbarwedge", "mathbin", "ogical and with horizontal dash")
addSymbol(0x02A5D, "midbarvee", "mathbin", "logical or with horizontal dash")
addSymbol(0x02A5E, "doublebarwedge", "mathbin", "logical and with double overbar")
addSymbol(0x02A5F, "wedgebar", "mathbin", "logical and with underbar")
addSymbol(0x02A60, "wedgedoublebar", "mathbin", "logical and with double underbar")
addSymbol(0x02A61, "varveebar", "mathbin", "small vee with underbar")
addSymbol(0x02A62, "doublebarvee", "mathbin", "logical or with double overbar")
addSymbol(0x02A63, "veedoublebar", "mathbin", "logical or with double underbar")
addSymbol(0x02A64, "dsub", "mathbin", "z notation domain antirestriction")
addSymbol(0x02A65, "rsub", "mathbin", "z notation range antirestriction")
addSymbol(0x02A66, "eqdot", "mathrel", "equals sign with dot below")
addSymbol(0x02A67, "dotequiv", "mathrel", "identical with dot above")
addSymbol(0x02A68, "equivVert", "mathrel", "triple horizontal bar with double vertical stroke")
addSymbol(0x02A69, "equivVvert", "mathrel", "triple horizontal bar with triple vertical stroke")
addSymbol(0x02A6A, "dotsim", "mathrel", "tilde operator with dot above")
addSymbol(0x02A6B, "simrdots", "mathrel", "tilde operator with rising dots")
addSymbol(0x02A6C, "simminussim", "mathrel", "similar minus similar")
addSymbol(0x02A6D, "congdot", "mathrel", "congruent with dot above")
addSymbol(0x02A6E, "asteq", "mathrel", "equals with asterisk")
addSymbol(0x02A6F, "hatapprox", "mathrel", "almost equal to with circumflex accent")
addSymbol(0x02A70, "approxeqq", "mathrel", "approximately equal or equal to")
addSymbol(0x02A71, "eqqplus", "mathbin", "equals sign above plus sign")
addSymbol(0x02A72, "pluseqq", "mathbin", "plus sign above equals sign")
addSymbol(0x02A73, "eqqsim", "mathrel", "equals sign above tilde operator")
addSymbol(0x02A74, "Coloneq", "mathrel", "double colon equal")
addSymbol(0x02A75, "eqeq", "mathrel", "two consecutive equals signs")
addSymbol(0x02A76, "eqeqeq", "mathrel", "three consecutive equals signs")
addSymbol(0x02A77, "ddotseq", "mathrel", "equals sign with two dots above and two dots below")
addSymbol(0x02A78, "equivDD", "mathrel", "equivalent with four dots above")
addSymbol(0x02A79, "ltcir", "mathrel", "less-than with circle inside")
addSymbol(0x02A7A, "gtcir", "mathrel", "greater-than with circle inside")
addSymbol(0x02A7B, "ltquest", "mathrel", "less-than with question mark above")
addSymbol(0x02A7C, "gtquest", "mathrel", "greater-than with question mark above")
addSymbol(0x02A7D, "leqslant", "mathrel", "less-than or slanted equal to")
addSymbol(0x02A7E, "geqslant", "mathrel", "greater-than or slanted equal to")
addSymbol(0x02A7F, "lesdot", "mathrel", "less-than or slanted equal to with dot inside")
addSymbol(0x02A80, "gesdot", "mathrel", "greater-than or slanted equal to with dot inside")
addSymbol(0x02A81, "lesdoto", "mathrel", "less-than or slanted equal to with dot above")
addSymbol(0x02A82, "gesdoto", "mathrel", "greater-than or slanted equal to with dot above")
addSymbol(0x02A83, "lesdotor", "mathrel", "less-than or slanted equal to with dot above right")
addSymbol(0x02A84, "gesdotol", "mathrel", "greater-than or slanted equal to with dot above left")
addSymbol(0x02A85, "lessapprox", "mathrel", "less-than or approximate")
addSymbol(0x02A86, "gtrapprox", "mathrel", "greater-than or approximate")
addSymbol(0x02A87, "lneq", "mathrel", "less-than and single-line not equal to")
addSymbol(0x02A88, "gneq", "mathrel", "greater-than and single-line not equal to")
addSymbol(0x02A89, "lnapprox", "mathrel", "less-than and not approximate")
addSymbol(0x02A8A, "gnapprox", "mathrel", "greater-than and not approximate")
addSymbol(0x02A8B, "lesseqqgtr", "mathrel", "less-than above double-line equal above greater-than")
addSymbol(0x02A8C, "gtreqqless", "mathrel", "greater-than above double-line equal above less-than")
addSymbol(0x02A8D, "lsime", "mathrel", "less-than above similar or equal")
addSymbol(0x02A8E, "gsime", "mathrel", "greater-than above similar or equal")
addSymbol(0x02A8F, "lsimg", "mathrel", "less-than above similar above greater-than")
addSymbol(0x02A90, "gsiml", "mathrel", "greater-than above similar above less-than")
addSymbol(0x02A91, "lgE", "mathrel", "less-than above greater-than above double-line equal")
addSymbol(0x02A92, "glE", "mathrel", "greater-than above less-than above double-line equal")
addSymbol(0x02A93, "lesges", "mathrel", "less-than above slanted equal above greater-than above slanted equal")
addSymbol(0x02A94, "gesles", "mathrel", "greater-than above slanted equal above less-than above slanted equal")
addSymbol(0x02A95, "eqslantless", "mathrel", "slanted equal to or less-than")
addSymbol(0x02A96, "eqslantgtr", "mathrel", "slanted equal to or greater-than")
addSymbol(0x02A97, "elsdot", "mathrel", "slanted equal to or less-than with dot inside")
addSymbol(0x02A98, "egsdot", "mathrel", "slanted equal to or greater-than with dot inside")
addSymbol(0x02A99, "eqqless", "mathrel", "double-line equal to or less-than")
addSymbol(0x02A9A, "eqqgtr", "mathrel", "double-line equal to or greater-than")
addSymbol(0x02A9B, "eqqslantless", "mathrel", "double-line slanted equal to or less-than")
addSymbol(0x02A9C, "eqqslantgtr", "mathrel", "double-line slanted equal to or greater-than")
addSymbol(0x02A9D, "simless", "mathrel", "similar or less-than")
addSymbol(0x02A9E, "simgtr", "mathrel", "similar or greater-than")
addSymbol(0x02A9F, "simlE", "mathrel", "similar above less-than above equals sign")
addSymbol(0x02AA0, "simgE", "mathrel", "similar above greater-than above equals sign")
addSymbol(0x02AA1, "Lt", "mathrel", "double nested less-than")
addSymbol(0x02AA2, "Gt", "mathrel", "double nested greater-than")
addSymbol(0x02AA3, "partialmeetcontraction", "mathrel", "double less-than with underbar")
addSymbol(0x02AA4, "glj", "mathrel", "greater-than overlapping less-than")
addSymbol(0x02AA5, "gla", "mathrel", "greater-than beside less-than")
addSymbol(0x02AA6, "ltcc", "mathrel", "less-than closed by curve")
addSymbol(0x02AA7, "gtcc", "mathrel", "greater-than closed by curve")
addSymbol(0x02AA8, "lescc", "mathrel", "less-than closed by curve above slanted equal")
addSymbol(0x02AA9, "gescc", "mathrel", "greater-than closed by curve above slanted equal")
addSymbol(0x02AAA, "smt", "mathrel", "smaller than")
addSymbol(0x02AAB, "lat", "mathrel", "larger than")
addSymbol(0x02AAC, "smte", "mathrel", "smaller than or equal to")
addSymbol(0x02AAD, "late", "mathrel", "larger than or equal to")
addSymbol(0x02AAE, "bumpeqq", "mathrel", "equals sign with bumpy above")
addSymbol(0x02AAF, "preceq", "mathrel", "precedes above single-line equals sign")
addSymbol(0x02AB0, "succeq", "mathrel", "succeeds above single-line equals sign")
addSymbol(0x02AB1, "precneq", "mathrel", "precedes above single-line not equal to")
addSymbol(0x02AB2, "succneq", "mathrel", "succeeds above single-line not equal to")
addSymbol(0x02AB3, "preceqq", "mathrel", "precedes above equals sign")
addSymbol(0x02AB4, "succeqq", "mathrel", "succeeds above equals sign")
addSymbol(0x02AB5, "precneqq", "mathrel", "precedes above not equal to")
addSymbol(0x02AB6, "succneqq", "mathrel", "succeeds above not equal to")
addSymbol(0x02AB7, "precapprox", "mathrel", "precedes above almost equal to")
addSymbol(0x02AB8, "succapprox", "mathrel", "succeeds above almost equal to")
addSymbol(0x02AB9, "precnapprox", "mathrel", "precedes above not almost equal to")
addSymbol(0x02ABA, "succnapprox", "mathrel", "succeeds above not almost equal to")
addSymbol(0x02ABB, "Prec", "mathrel", "double precedes")
addSymbol(0x02ABC, "Succ", "mathrel", "double succeeds")
addSymbol(0x02ABD, "subsetdot", "mathrel", "subset with dot")
addSymbol(0x02ABE, "supsetdot", "mathrel", "superset with dot")
addSymbol(0x02ABF, "subsetplus", "mathrel", "subset with plus sign below")
addSymbol(0x02AC0, "supsetplus", "mathrel", "superset with plus sign below")
addSymbol(0x02AC1, "submult", "mathrel", "subset with multiplication sign below")
addSymbol(0x02AC2, "supmult", "mathrel", "superset with multiplication sign below")
addSymbol(0x02AC3, "subedot", "mathrel", "subset of or equal to with dot above")
addSymbol(0x02AC4, "supedot", "mathrel", "superset of or equal to with dot above")
addSymbol(0x02AC5, "subseteqq", "mathrel", "subset of above equals sign")
addSymbol(0x02AC6, "supseteqq", "mathrel", "superset of above equals sign")
addSymbol(0x02AC7, "subsim", "mathrel", "subset of above tilde operator")
addSymbol(0x02AC8, "supsim", "mathrel", "superset of above tilde operator")
addSymbol(0x02AC9, "subsetapprox", "mathrel", "subset of above almost equal to")
addSymbol(0x02ACA, "supsetapprox", "mathrel", "superset of above almost equal to")
addSymbol(0x02ACB, "subsetneqq", "mathrel", "subset of above not equal to")
addSymbol(0x02ACC, "supsetneqq", "mathrel", "superset of above not equal to")
addSymbol(0x02ACD, "lsqhook", "mathrel", "square left open box operator")
addSymbol(0x02ACE, "rsqhook", "mathrel", "square right open box operator")
addSymbol(0x02ACF, "csub", "mathrel", "closed subset")
addSymbol(0x02AD0, "csup", "mathrel", "closed superset")
addSymbol(0x02AD1, "csube", "mathrel", "closed subset or equal to")
addSymbol(0x02AD2, "csupe", "mathrel", "closed superset or equal to")
addSymbol(0x02AD3, "subsup", "mathrel", "subset above superset")
addSymbol(0x02AD4, "supsub", "mathrel", "superset above subset")
addSymbol(0x02AD5, "subsub", "mathrel", "subset above subset")
addSymbol(0x02AD6, "supsup", "mathrel", "superset above superset")
addSymbol(0x02AD7, "suphsub", "mathrel", "superset beside subset")
addSymbol(0x02AD8, "supdsub", "mathrel", "superset beside and joined by dash with subset")
addSymbol(0x02AD9, "forkv", "mathrel", "element of opening downwards")
addSymbol(0x02ADA, "topfork", "mathrel", "pitchfork with tee top")
addSymbol(0x02ADB, "mlcp", "mathrel", "transversal intersection")
addSymbol(0x02ADC, "forks", "mathrel", "forking")
addSymbol(0x02ADD, "forksnot", "mathrel", "nonforking")
addSymbol(0x02ADE, "shortlefttack", "mathrel", "short left tack")
addSymbol(0x02ADF, "shortdowntack", "mathrel", "short down tack")
addSymbol(0x02AE0, "shortuptack", "mathrel", "short up tack")
addSymbol(0x02AE1, "perps", "mathord", "perpendicular with s")
addSymbol(0x02AE2, "vDdash", "mathrel", "vertical bar triple right turnstile")
addSymbol(0x02AE3, "dashV", "mathrel", "double vertical bar left turnstile")
addSymbol(0x02AE4, "Dashv", "mathrel", "vertical bar double left turnstile")
addSymbol(0x02AE5, "DashV", "mathrel", "double vertical bar double left turnstile")
addSymbol(0x02AE6, "varVdash", "mathrel", "long dash from left member of double vertical")
addSymbol(0x02AE7, "Barv", "mathrel", "short down tack with overbar")
addSymbol(0x02AE8, "vBar", "mathrel", "short up tack with underbar")
addSymbol(0x02AE9, "vBarv", "mathrel", "short up tack above short down tack")
addSymbol(0x02AEA, "barV", "mathrel", "double down tack")
addSymbol(0x02AEB, "Vbar", "mathrel", "double up tack")
addSymbol(0x02AEC, "Not", "mathrel", "double stroke not sign")
addSymbol(0x02AED, "bNot", "mathrel", "reversed double stroke not sign")
addSymbol(0x02AEE, "revnmid", "mathrel", "does not divide with reversed negation slash")
addSymbol(0x02AEF, "cirmid", "mathrel", "vertical line with circle above")
addSymbol(0x02AF0, "midcir", "mathrel", "vertical line with circle below")
addSymbol(0x02AF1, "topcir", "mathord", "down tack with circle below")
addSymbol(0x02AF2, "nhpar", "mathrel", "parallel with horizontal stroke")
addSymbol(0x02AF3, "parsim", "mathrel", "parallel with tilde operator")
addSymbol(0x02AF4, "interleave", "mathbin", "triple vertical bar binary relation")
addSymbol(0x02AF5, "nhVvert", "mathbin", "triple vertical bar with horizontal stroke")
addSymbol(0x02AF6, "threedotcolon", "mathbin", "triple colon operator")
addSymbol(0x02AF7, "lllnest", "mathrel", "stacked very much less-than")
addSymbol(0x02AF8, "gggnest", "mathrel", "stacked very much greater-than")
addSymbol(0x02AF9, "leqqslant", "mathrel", "double-line slanted less-than or equal to")
addSymbol(0x02AFA, "geqqslant", "mathrel", "double-line slanted greater-than or equal to")
addSymbol(0x02AFB, "trslash", "mathbin", "triple solidus binary relation")
addSymbol(0x02AFC, "biginterleave", "mathop", "large triple vertical bar operator")
addSymbol(0x02AFD, "sslash", "mathbin", "double solidus operator")
addSymbol(0x02AFE, "talloblong", "mathbin", "white vertical bar")
addSymbol(0x02AFF, "bigtalloblong", "mathop", "n-ary white vertical bar")
addSymbol(0x02B12, "squaretopblack", "mathord", "square with top half black")
addSymbol(0x02B13, "squarebotblack", "mathord", "square with bottom half black")
addSymbol(0x02B14, "squareurblack", "mathord", "square with upper right diagonal half black")
addSymbol(0x02B15, "squarellblack", "mathord", "square with lower left diagonal half black")
addSymbol(0x02B16, "diamondleftblack", "mathord", "diamond with left half black")
addSymbol(0x02B17, "diamondrightblack", "mathord", "diamond with right half black")
addSymbol(0x02B18, "diamondtopblack", "mathord", "diamond with top half black")
addSymbol(0x02B19, "diamondbotblack", "mathord", "diamond with bottom half black")
addSymbol(0x02B1A, "dottedsquare", "mathord", "dotted square")
addSymbol(0x02B1B, "lgblksquare", "mathord", "black large square")
addSymbol(0x02B1C, "lgwhtsquare", "mathord", "white large square")
addSymbol(0x02B1D, "vysmblksquare", "mathord", "black very small square")
addSymbol(0x02B1E, "vysmwhtsquare", "mathord", "white very small square")
addSymbol(0x02B1F, "pentagonblack", "mathord", "black pentagon")
addSymbol(0x02B20, "pentagon", "mathord", "white pentagon")
addSymbol(0x02B21, "varhexagon", "mathord", "white hexagon")
addSymbol(0x02B22, "varhexagonblack", "mathord", "black hexagon")
addSymbol(0x02B23, "hexagonblack", "mathord", "horizontal black hexagon")
addSymbol(0x02B24, "lgblkcircle", "mathord", "black large circle")
addSymbol(0x02B25, "mdblkdiamond", "mathord", "black medium diamond")
addSymbol(0x02B26, "mdwhtdiamond", "mathord", "white medium diamond")
addSymbol(0x02B27, "mdblklozenge", "mathord", "black medium lozenge")
addSymbol(0x02B28, "mdwhtlozenge", "mathord", "white medium lozenge")
addSymbol(0x02B29, "smblkdiamond", "mathord", "black small diamond")
addSymbol(0x02B2A, "smblklozenge", "mathord", "black small lozenge")
addSymbol(0x02B2B, "smwhtlozenge", "mathord", "white small lozenge")
addSymbol(0x02B2C, "blkhorzoval", "mathord", "black horizontal ellipse")
addSymbol(0x02B2D, "whthorzoval", "mathord", "white horizontal ellipse")
addSymbol(0x02B2E, "blkvertoval", "mathord", "black vertical ellipse")
addSymbol(0x02B2F, "whtvertoval", "mathord", "white vertical ellipse")
addSymbol(0x02B30, "circleonleftarrow", "mathrel", "left arrow with small circle")
addSymbol(0x02B31, "leftthreearrows", "mathrel", "three leftwards arrows")
addSymbol(0x02B32, "leftarrowonoplus", "mathrel", "left arrow with circled plus")
addSymbol(0x02B33, "longleftsquigarrow", "mathrel", "long leftwards squiggle arrow")
addSymbol(0x02B34, "nvtwoheadleftarrow", "mathrel", "leftwards two-headed arrow with vertical stroke")
addSymbol(0x02B35, "nVtwoheadleftarrow", "mathrel", "leftwards two-headed arrow with double vertical stroke")
addSymbol(0x02B36, "twoheadmapsfrom", "mathrel", "leftwards two-headed arrow from bar")
addSymbol(0x02B37, "twoheadleftdbkarrow", "mathrel", "leftwards two-headed triple-dash arrow")
addSymbol(0x02B38, "leftdotarrow", "mathrel", "leftwards arrow with dotted stem")
addSymbol(0x02B39, "nvleftarrowtail", "mathrel", "leftwards arrow with tail with vertical stroke")
addSymbol(0x02B3A, "nVleftarrowtail", "mathrel", "leftwards arrow with tail with double vertical stroke")
addSymbol(0x02B3B, "twoheadleftarrowtail", "mathrel", "leftwards two-headed arrow with tail")
addSymbol(0x02B3C, "nvtwoheadleftarrowtail", "mathrel", "leftwards two-headed arrow with tail with vertical stroke")
addSymbol(
   0x02B3D,
   "nVtwoheadleftarrowtail",
   "mathrel",
   "leftwards two-headed arrow with tail with double vertical stroke"
)
addSymbol(0x02B3E, "leftarrowx", "mathrel", "leftwards arrow through x")
addSymbol(0x02B3F, "leftcurvedarrow", "mathrel", "wave arrow pointing directly left")
addSymbol(0x02B40, "equalleftarrow", "mathrel", "equals sign above leftwards arrow")
addSymbol(0x02B41, "bsimilarleftarrow", "mathrel", "reverse tilde operator above leftwards arrow")
addSymbol(0x02B42, "leftarrowbackapprox", "mathrel", "leftwards arrow above reverse almost equal to")
addSymbol(0x02B43, "rightarrowgtr", "mathrel", "rightwards arrow through greater-than")
addSymbol(0x02B44, "rightarrowsupset", "mathrel", "rightwards arrow through subset")
addSymbol(0x02B45, "LLeftarrow", "mathrel", "leftwards quadruple arrow")
addSymbol(0x02B46, "RRightarrow", "mathrel", "rightwards quadruple arrow")
addSymbol(0x02B47, "bsimilarrightarrow", "mathrel", "reverse tilde operator above rightwards arrow")
addSymbol(0x02B48, "rightarrowbackapprox", "mathrel", "rightwards arrow above reverse almost equal to")
addSymbol(0x02B49, "similarleftarrow", "mathrel", "tilde operator above leftwards arrow")
addSymbol(0x02B4A, "leftarrowapprox", "mathrel", "leftwards arrow above almost equal to")
addSymbol(0x02B4B, "leftarrowbsimilar", "mathrel", "leftwards arrow above reverse tilde operator")
addSymbol(0x02B4C, "rightarrowbsimilar", "mathrel", "righttwards arrow above reverse tilde operator")
addSymbol(0x02B50, "medwhitestar", "mathord", "white medium star")
addSymbol(0x02B51, "medblackstar", "mathord", "black medium star")
addSymbol(0x02B52, "smwhitestar", "mathord", "white small star")
addSymbol(0x02B53, "rightpentagonblack", "mathord", "black right-pointing pentagon")
addSymbol(0x02B54, "rightpentagon", "mathord", "white right-pointing pentagon")
addSymbol(0x03012, "postalmark", "mathord", "postal mark")
addSymbol(0x03030, "hzigzag", "mathord", "zigzag")
addSymbol(0x1D400, "mbfA", "mathalpha", "mathematical bold capital a")
addSymbol(0x1D401, "mbfB", "mathalpha", "mathematical bold capital b")
addSymbol(0x1D402, "mbfC", "mathalpha", "mathematical bold capital c")
addSymbol(0x1D403, "mbfD", "mathalpha", "mathematical bold capital d")
addSymbol(0x1D404, "mbfE", "mathalpha", "mathematical bold capital e")
addSymbol(0x1D405, "mbfF", "mathalpha", "mathematical bold capital f")
addSymbol(0x1D406, "mbfG", "mathalpha", "mathematical bold capital g")
addSymbol(0x1D407, "mbfH", "mathalpha", "mathematical bold capital h")
addSymbol(0x1D408, "mbfI", "mathalpha", "mathematical bold capital i")
addSymbol(0x1D409, "mbfJ", "mathalpha", "mathematical bold capital j")
addSymbol(0x1D40A, "mbfK", "mathalpha", "mathematical bold capital k")
addSymbol(0x1D40B, "mbfL", "mathalpha", "mathematical bold capital l")
addSymbol(0x1D40C, "mbfM", "mathalpha", "mathematical bold capital m")
addSymbol(0x1D40D, "mbfN", "mathalpha", "mathematical bold capital n")
addSymbol(0x1D40E, "mbfO", "mathalpha", "mathematical bold capital o")
addSymbol(0x1D40F, "mbfP", "mathalpha", "mathematical bold capital p")
addSymbol(0x1D410, "mbfQ", "mathalpha", "mathematical bold capital q")
addSymbol(0x1D411, "mbfR", "mathalpha", "mathematical bold capital r")
addSymbol(0x1D412, "mbfS", "mathalpha", "mathematical bold capital s")
addSymbol(0x1D413, "mbfT", "mathalpha", "mathematical bold capital t")
addSymbol(0x1D414, "mbfU", "mathalpha", "mathematical bold capital u")
addSymbol(0x1D415, "mbfV", "mathalpha", "mathematical bold capital v")
addSymbol(0x1D416, "mbfW", "mathalpha", "mathematical bold capital w")
addSymbol(0x1D417, "mbfX", "mathalpha", "mathematical bold capital x")
addSymbol(0x1D418, "mbfY", "mathalpha", "mathematical bold capital y")
addSymbol(0x1D419, "mbfZ", "mathalpha", "mathematical bold capital z")
addSymbol(0x1D41A, "mbfa", "mathalpha", "mathematical bold small a")
addSymbol(0x1D41B, "mbfb", "mathalpha", "mathematical bold small b")
addSymbol(0x1D41C, "mbfc", "mathalpha", "mathematical bold small c")
addSymbol(0x1D41D, "mbfd", "mathalpha", "mathematical bold small d")
addSymbol(0x1D41E, "mbfe", "mathalpha", "mathematical bold small e")
addSymbol(0x1D41F, "mbff", "mathalpha", "mathematical bold small f")
addSymbol(0x1D420, "mbfg", "mathalpha", "mathematical bold small g")
addSymbol(0x1D421, "mbfh", "mathalpha", "mathematical bold small h")
addSymbol(0x1D422, "mbfi", "mathalpha", "mathematical bold small i")
addSymbol(0x1D423, "mbfj", "mathalpha", "mathematical bold small j")
addSymbol(0x1D424, "mbfk", "mathalpha", "mathematical bold small k")
addSymbol(0x1D425, "mbfl", "mathalpha", "mathematical bold small l")
addSymbol(0x1D426, "mbfm", "mathalpha", "mathematical bold small m")
addSymbol(0x1D427, "mbfn", "mathalpha", "mathematical bold small n")
addSymbol(0x1D428, "mbfo", "mathalpha", "mathematical bold small o")
addSymbol(0x1D429, "mbfp", "mathalpha", "mathematical bold small p")
addSymbol(0x1D42A, "mbfq", "mathalpha", "mathematical bold small q")
addSymbol(0x1D42B, "mbfr", "mathalpha", "mathematical bold small r")
addSymbol(0x1D42C, "mbfs", "mathalpha", "mathematical bold small s")
addSymbol(0x1D42D, "mbft", "mathalpha", "mathematical bold small t")
addSymbol(0x1D42E, "mbfu", "mathalpha", "mathematical bold small u")
addSymbol(0x1D42F, "mbfv", "mathalpha", "mathematical bold small v")
addSymbol(0x1D430, "mbfw", "mathalpha", "mathematical bold small w")
addSymbol(0x1D431, "mbfx", "mathalpha", "mathematical bold small x")
addSymbol(0x1D432, "mbfy", "mathalpha", "mathematical bold small y")
addSymbol(0x1D433, "mbfz", "mathalpha", "mathematical bold small z")
addSymbol(0x1D434, "mitA", "mathalpha", "mathematical italic capital a")
addSymbol(0x1D435, "mitB", "mathalpha", "mathematical italic capital b")
addSymbol(0x1D436, "mitC", "mathalpha", "mathematical italic capital c")
addSymbol(0x1D437, "mitD", "mathalpha", "mathematical italic capital d")
addSymbol(0x1D438, "mitE", "mathalpha", "mathematical italic capital e")
addSymbol(0x1D439, "mitF", "mathalpha", "mathematical italic capital f")
addSymbol(0x1D43A, "mitG", "mathalpha", "mathematical italic capital g")
addSymbol(0x1D43B, "mitH", "mathalpha", "mathematical italic capital h")
addSymbol(0x1D43C, "mitI", "mathalpha", "mathematical italic capital i")
addSymbol(0x1D43D, "mitJ", "mathalpha", "mathematical italic capital j")
addSymbol(0x1D43E, "mitK", "mathalpha", "mathematical italic capital k")
addSymbol(0x1D43F, "mitL", "mathalpha", "mathematical italic capital l")
addSymbol(0x1D440, "mitM", "mathalpha", "mathematical italic capital m")
addSymbol(0x1D441, "mitN", "mathalpha", "mathematical italic capital n")
addSymbol(0x1D442, "mitO", "mathalpha", "mathematical italic capital o")
addSymbol(0x1D443, "mitP", "mathalpha", "mathematical italic capital p")
addSymbol(0x1D444, "mitQ", "mathalpha", "mathematical italic capital q")
addSymbol(0x1D445, "mitR", "mathalpha", "mathematical italic capital r")
addSymbol(0x1D446, "mitS", "mathalpha", "mathematical italic capital s")
addSymbol(0x1D447, "mitT", "mathalpha", "mathematical italic capital t")
addSymbol(0x1D448, "mitU", "mathalpha", "mathematical italic capital u")
addSymbol(0x1D449, "mitV", "mathalpha", "mathematical italic capital v")
addSymbol(0x1D44A, "mitW", "mathalpha", "mathematical italic capital w")
addSymbol(0x1D44B, "mitX", "mathalpha", "mathematical italic capital x")
addSymbol(0x1D44C, "mitY", "mathalpha", "mathematical italic capital y")
addSymbol(0x1D44D, "mitZ", "mathalpha", "mathematical italic capital z")
addSymbol(0x1D44E, "mita", "mathalpha", "mathematical italic small a")
addSymbol(0x1D44F, "mitb", "mathalpha", "mathematical italic small b")
addSymbol(0x1D450, "mitc", "mathalpha", "mathematical italic small c")
addSymbol(0x1D451, "mitd", "mathalpha", "mathematical italic small d")
addSymbol(0x1D452, "mite", "mathalpha", "mathematical italic small e")
addSymbol(0x1D453, "mitf", "mathalpha", "mathematical italic small f")
addSymbol(0x1D454, "mitg", "mathalpha", "mathematical italic small g")
addSymbol(0x1D456, "miti", "mathalpha", "mathematical italic small i")
addSymbol(0x1D457, "mitj", "mathalpha", "mathematical italic small j")
addSymbol(0x1D458, "mitk", "mathalpha", "mathematical italic small k")
addSymbol(0x1D459, "mitl", "mathalpha", "mathematical italic small l")
addSymbol(0x1D45A, "mitm", "mathalpha", "mathematical italic small m")
addSymbol(0x1D45B, "mitn", "mathalpha", "mathematical italic small n")
addSymbol(0x1D45C, "mito", "mathalpha", "mathematical italic small o")
addSymbol(0x1D45D, "mitp", "mathalpha", "mathematical italic small p")
addSymbol(0x1D45E, "mitq", "mathalpha", "mathematical italic small q")
addSymbol(0x1D45F, "mitr", "mathalpha", "mathematical italic small r")
addSymbol(0x1D460, "mits", "mathalpha", "mathematical italic small s")
addSymbol(0x1D461, "mitt", "mathalpha", "mathematical italic small t")
addSymbol(0x1D462, "mitu", "mathalpha", "mathematical italic small u")
addSymbol(0x1D463, "mitv", "mathalpha", "mathematical italic small v")
addSymbol(0x1D464, "mitw", "mathalpha", "mathematical italic small w")
addSymbol(0x1D465, "mitx", "mathalpha", "mathematical italic small x")
addSymbol(0x1D466, "mity", "mathalpha", "mathematical italic small y")
addSymbol(0x1D467, "mitz", "mathalpha", "mathematical italic small z")
addSymbol(0x1D468, "mbfitA", "mathalpha", "mathematical bold italic capital a")
addSymbol(0x1D469, "mbfitB", "mathalpha", "mathematical bold italic capital b")
addSymbol(0x1D46A, "mbfitC", "mathalpha", "mathematical bold italic capital c")
addSymbol(0x1D46B, "mbfitD", "mathalpha", "mathematical bold italic capital d")
addSymbol(0x1D46C, "mbfitE", "mathalpha", "mathematical bold italic capital e")
addSymbol(0x1D46D, "mbfitF", "mathalpha", "mathematical bold italic capital f")
addSymbol(0x1D46E, "mbfitG", "mathalpha", "mathematical bold italic capital g")
addSymbol(0x1D46F, "mbfitH", "mathalpha", "mathematical bold italic capital h")
addSymbol(0x1D470, "mbfitI", "mathalpha", "mathematical bold italic capital i")
addSymbol(0x1D471, "mbfitJ", "mathalpha", "mathematical bold italic capital j")
addSymbol(0x1D472, "mbfitK", "mathalpha", "mathematical bold italic capital k")
addSymbol(0x1D473, "mbfitL", "mathalpha", "mathematical bold italic capital l")
addSymbol(0x1D474, "mbfitM", "mathalpha", "mathematical bold italic capital m")
addSymbol(0x1D475, "mbfitN", "mathalpha", "mathematical bold italic capital n")
addSymbol(0x1D476, "mbfitO", "mathalpha", "mathematical bold italic capital o")
addSymbol(0x1D477, "mbfitP", "mathalpha", "mathematical bold italic capital p")
addSymbol(0x1D478, "mbfitQ", "mathalpha", "mathematical bold italic capital q")
addSymbol(0x1D479, "mbfitR", "mathalpha", "mathematical bold italic capital r")
addSymbol(0x1D47A, "mbfitS", "mathalpha", "mathematical bold italic capital s")
addSymbol(0x1D47B, "mbfitT", "mathalpha", "mathematical bold italic capital t")
addSymbol(0x1D47C, "mbfitU", "mathalpha", "mathematical bold italic capital u")
addSymbol(0x1D47D, "mbfitV", "mathalpha", "mathematical bold italic capital v")
addSymbol(0x1D47E, "mbfitW", "mathalpha", "mathematical bold italic capital w")
addSymbol(0x1D47F, "mbfitX", "mathalpha", "mathematical bold italic capital x")
addSymbol(0x1D480, "mbfitY", "mathalpha", "mathematical bold italic capital y")
addSymbol(0x1D481, "mbfitZ", "mathalpha", "mathematical bold italic capital z")
addSymbol(0x1D482, "mbfita", "mathalpha", "mathematical bold italic small a")
addSymbol(0x1D483, "mbfitb", "mathalpha", "mathematical bold italic small b")
addSymbol(0x1D484, "mbfitc", "mathalpha", "mathematical bold italic small c")
addSymbol(0x1D485, "mbfitd", "mathalpha", "mathematical bold italic small d")
addSymbol(0x1D486, "mbfite", "mathalpha", "mathematical bold italic small e")
addSymbol(0x1D487, "mbfitf", "mathalpha", "mathematical bold italic small f")
addSymbol(0x1D488, "mbfitg", "mathalpha", "mathematical bold italic small g")
addSymbol(0x1D489, "mbfith", "mathalpha", "mathematical bold italic small h")
addSymbol(0x1D48A, "mbfiti", "mathalpha", "mathematical bold italic small i")
addSymbol(0x1D48B, "mbfitj", "mathalpha", "mathematical bold italic small j")
addSymbol(0x1D48C, "mbfitk", "mathalpha", "mathematical bold italic small k")
addSymbol(0x1D48D, "mbfitl", "mathalpha", "mathematical bold italic small l")
addSymbol(0x1D48E, "mbfitm", "mathalpha", "mathematical bold italic small m")
addSymbol(0x1D48F, "mbfitn", "mathalpha", "mathematical bold italic small n")
addSymbol(0x1D490, "mbfito", "mathalpha", "mathematical bold italic small o")
addSymbol(0x1D491, "mbfitp", "mathalpha", "mathematical bold italic small p")
addSymbol(0x1D492, "mbfitq", "mathalpha", "mathematical bold italic small q")
addSymbol(0x1D493, "mbfitr", "mathalpha", "mathematical bold italic small r")
addSymbol(0x1D494, "mbfits", "mathalpha", "mathematical bold italic small s")
addSymbol(0x1D495, "mbfitt", "mathalpha", "mathematical bold italic small t")
addSymbol(0x1D496, "mbfitu", "mathalpha", "mathematical bold italic small u")
addSymbol(0x1D497, "mbfitv", "mathalpha", "mathematical bold italic small v")
addSymbol(0x1D498, "mbfitw", "mathalpha", "mathematical bold italic small w")
addSymbol(0x1D499, "mbfitx", "mathalpha", "mathematical bold italic small x")
addSymbol(0x1D49A, "mbfity", "mathalpha", "mathematical bold italic small y")
addSymbol(0x1D49B, "mbfitz", "mathalpha", "mathematical bold italic small z")
addSymbol(0x1D49C, "mscrA", "mathalpha", "mathematical script capital a")
addSymbol(0x1D49E, "mscrC", "mathalpha", "mathematical script capital c")
addSymbol(0x1D49F, "mscrD", "mathalpha", "mathematical script capital d")
addSymbol(0x1D4A2, "mscrG", "mathalpha", "mathematical script capital g")
addSymbol(0x1D4A5, "mscrJ", "mathalpha", "mathematical script capital j")
addSymbol(0x1D4A6, "mscrK", "mathalpha", "mathematical script capital k")
addSymbol(0x1D4A9, "mscrN", "mathalpha", "mathematical script capital n")
addSymbol(0x1D4AA, "mscrO", "mathalpha", "mathematical script capital o")
addSymbol(0x1D4AB, "mscrP", "mathalpha", "mathematical script capital p")
addSymbol(0x1D4AC, "mscrQ", "mathalpha", "mathematical script capital q")
addSymbol(0x1D4AE, "mscrS", "mathalpha", "mathematical script capital s")
addSymbol(0x1D4AF, "mscrT", "mathalpha", "mathematical script capital t")
addSymbol(0x1D4B0, "mscrU", "mathalpha", "mathematical script capital u")
addSymbol(0x1D4B1, "mscrV", "mathalpha", "mathematical script capital v")
addSymbol(0x1D4B2, "mscrW", "mathalpha", "mathematical script capital w")
addSymbol(0x1D4B3, "mscrX", "mathalpha", "mathematical script capital x")
addSymbol(0x1D4B4, "mscrY", "mathalpha", "mathematical script capital y")
addSymbol(0x1D4B5, "mscrZ", "mathalpha", "mathematical script capital z")
addSymbol(0x1D4B6, "mscra", "mathalpha", "mathematical script small a")
addSymbol(0x1D4B7, "mscrb", "mathalpha", "mathematical script small b")
addSymbol(0x1D4B8, "mscrc", "mathalpha", "mathematical script small c")
addSymbol(0x1D4B9, "mscrd", "mathalpha", "mathematical script small d")
addSymbol(0x1D4BB, "mscrf", "mathalpha", "mathematical script small f")
addSymbol(0x1D4BD, "mscrh", "mathalpha", "mathematical script small h")
addSymbol(0x1D4BE, "mscri", "mathalpha", "mathematical script small i")
addSymbol(0x1D4BF, "mscrj", "mathalpha", "mathematical script small j")
addSymbol(0x1D4C0, "mscrk", "mathalpha", "mathematical script small k")
addSymbol(0x1D4C1, "mscrl", "mathalpha", "mathematical script small l")
addSymbol(0x1D4C2, "mscrm", "mathalpha", "mathematical script small m")
addSymbol(0x1D4C3, "mscrn", "mathalpha", "mathematical script small n")
addSymbol(0x1D4C5, "mscrp", "mathalpha", "mathematical script small p")
addSymbol(0x1D4C6, "mscrq", "mathalpha", "mathematical script small q")
addSymbol(0x1D4C7, "mscrr", "mathalpha", "mathematical script small r")
addSymbol(0x1D4C8, "mscrs", "mathalpha", "mathematical script small s")
addSymbol(0x1D4C9, "mscrt", "mathalpha", "mathematical script small t")
addSymbol(0x1D4CA, "mscru", "mathalpha", "mathematical script small u")
addSymbol(0x1D4CB, "mscrv", "mathalpha", "mathematical script small v")
addSymbol(0x1D4CC, "mscrw", "mathalpha", "mathematical script small w")
addSymbol(0x1D4CD, "mscrx", "mathalpha", "mathematical script small x")
addSymbol(0x1D4CE, "mscry", "mathalpha", "mathematical script small y")
addSymbol(0x1D4CF, "mscrz", "mathalpha", "mathematical script small z")
addSymbol(0x1D4D0, "mbfscrA", "mathalpha", "mathematical bold script capital a")
addSymbol(0x1D4D1, "mbfscrB", "mathalpha", "mathematical bold script capital b")
addSymbol(0x1D4D2, "mbfscrC", "mathalpha", "mathematical bold script capital c")
addSymbol(0x1D4D3, "mbfscrD", "mathalpha", "mathematical bold script capital d")
addSymbol(0x1D4D4, "mbfscrE", "mathalpha", "mathematical bold script capital e")
addSymbol(0x1D4D5, "mbfscrF", "mathalpha", "mathematical bold script capital f")
addSymbol(0x1D4D6, "mbfscrG", "mathalpha", "mathematical bold script capital g")
addSymbol(0x1D4D7, "mbfscrH", "mathalpha", "mathematical bold script capital h")
addSymbol(0x1D4D8, "mbfscrI", "mathalpha", "mathematical bold script capital i")
addSymbol(0x1D4D9, "mbfscrJ", "mathalpha", "mathematical bold script capital j")
addSymbol(0x1D4DA, "mbfscrK", "mathalpha", "mathematical bold script capital k")
addSymbol(0x1D4DB, "mbfscrL", "mathalpha", "mathematical bold script capital l")
addSymbol(0x1D4DC, "mbfscrM", "mathalpha", "mathematical bold script capital m")
addSymbol(0x1D4DD, "mbfscrN", "mathalpha", "mathematical bold script capital n")
addSymbol(0x1D4DE, "mbfscrO", "mathalpha", "mathematical bold script capital o")
addSymbol(0x1D4DF, "mbfscrP", "mathalpha", "mathematical bold script capital p")
addSymbol(0x1D4E0, "mbfscrQ", "mathalpha", "mathematical bold script capital q")
addSymbol(0x1D4E1, "mbfscrR", "mathalpha", "mathematical bold script capital r")
addSymbol(0x1D4E2, "mbfscrS", "mathalpha", "mathematical bold script capital s")
addSymbol(0x1D4E3, "mbfscrT", "mathalpha", "mathematical bold script capital t")
addSymbol(0x1D4E4, "mbfscrU", "mathalpha", "mathematical bold script capital u")
addSymbol(0x1D4E5, "mbfscrV", "mathalpha", "mathematical bold script capital v")
addSymbol(0x1D4E6, "mbfscrW", "mathalpha", "mathematical bold script capital w")
addSymbol(0x1D4E7, "mbfscrX", "mathalpha", "mathematical bold script capital x")
addSymbol(0x1D4E8, "mbfscrY", "mathalpha", "mathematical bold script capital y")
addSymbol(0x1D4E9, "mbfscrZ", "mathalpha", "mathematical bold script capital z")
addSymbol(0x1D4EA, "mbfscra", "mathalpha", "mathematical bold script small a")
addSymbol(0x1D4EB, "mbfscrb", "mathalpha", "mathematical bold script small b")
addSymbol(0x1D4EC, "mbfscrc", "mathalpha", "mathematical bold script small c")
addSymbol(0x1D4ED, "mbfscrd", "mathalpha", "mathematical bold script small d")
addSymbol(0x1D4EE, "mbfscre", "mathalpha", "mathematical bold script small e")
addSymbol(0x1D4EF, "mbfscrf", "mathalpha", "mathematical bold script small f")
addSymbol(0x1D4F0, "mbfscrg", "mathalpha", "mathematical bold script small g")
addSymbol(0x1D4F1, "mbfscrh", "mathalpha", "mathematical bold script small h")
addSymbol(0x1D4F2, "mbfscri", "mathalpha", "mathematical bold script small i")
addSymbol(0x1D4F3, "mbfscrj", "mathalpha", "mathematical bold script small j")
addSymbol(0x1D4F4, "mbfscrk", "mathalpha", "mathematical bold script small k")
addSymbol(0x1D4F5, "mbfscrl", "mathalpha", "mathematical bold script small l")
addSymbol(0x1D4F6, "mbfscrm", "mathalpha", "mathematical bold script small m")
addSymbol(0x1D4F7, "mbfscrn", "mathalpha", "mathematical bold script small n")
addSymbol(0x1D4F8, "mbfscro", "mathalpha", "mathematical bold script small o")
addSymbol(0x1D4F9, "mbfscrp", "mathalpha", "mathematical bold script small p")
addSymbol(0x1D4FA, "mbfscrq", "mathalpha", "mathematical bold script small q")
addSymbol(0x1D4FB, "mbfscrr", "mathalpha", "mathematical bold script small r")
addSymbol(0x1D4FC, "mbfscrs", "mathalpha", "mathematical bold script small s")
addSymbol(0x1D4FD, "mbfscrt", "mathalpha", "mathematical bold script small t")
addSymbol(0x1D4FE, "mbfscru", "mathalpha", "mathematical bold script small u")
addSymbol(0x1D4FF, "mbfscrv", "mathalpha", "mathematical bold script small v")
addSymbol(0x1D500, "mbfscrw", "mathalpha", "mathematical bold script small w")
addSymbol(0x1D501, "mbfscrx", "mathalpha", "mathematical bold script small x")
addSymbol(0x1D502, "mbfscry", "mathalpha", "mathematical bold script small y")
addSymbol(0x1D503, "mbfscrz", "mathalpha", "mathematical bold script small z")
addSymbol(0x1D504, "mfrakA", "mathalpha", "mathematical fraktur capital a")
addSymbol(0x1D505, "mfrakB", "mathalpha", "mathematical fraktur capital b")
addSymbol(0x1D507, "mfrakD", "mathalpha", "mathematical fraktur capital d")
addSymbol(0x1D508, "mfrakE", "mathalpha", "mathematical fraktur capital e")
addSymbol(0x1D509, "mfrakF", "mathalpha", "mathematical fraktur capital f")
addSymbol(0x1D50A, "mfrakG", "mathalpha", "mathematical fraktur capital g")
addSymbol(0x1D50D, "mfrakJ", "mathalpha", "mathematical fraktur capital j")
addSymbol(0x1D50E, "mfrakK", "mathalpha", "mathematical fraktur capital k")
addSymbol(0x1D50F, "mfrakL", "mathalpha", "mathematical fraktur capital l")
addSymbol(0x1D510, "mfrakM", "mathalpha", "mathematical fraktur capital m")
addSymbol(0x1D511, "mfrakN", "mathalpha", "mathematical fraktur capital n")
addSymbol(0x1D512, "mfrakO", "mathalpha", "mathematical fraktur capital o")
addSymbol(0x1D513, "mfrakP", "mathalpha", "mathematical fraktur capital p")
addSymbol(0x1D514, "mfrakQ", "mathalpha", "mathematical fraktur capital q")
addSymbol(0x1D516, "mfrakS", "mathalpha", "mathematical fraktur capital s")
addSymbol(0x1D517, "mfrakT", "mathalpha", "mathematical fraktur capital t")
addSymbol(0x1D518, "mfrakU", "mathalpha", "mathematical fraktur capital u")
addSymbol(0x1D519, "mfrakV", "mathalpha", "mathematical fraktur capital v")
addSymbol(0x1D51A, "mfrakW", "mathalpha", "mathematical fraktur capital w")
addSymbol(0x1D51B, "mfrakX", "mathalpha", "mathematical fraktur capital x")
addSymbol(0x1D51C, "mfrakY", "mathalpha", "mathematical fraktur capital y")
addSymbol(0x1D51E, "mfraka", "mathalpha", "mathematical fraktur small a")
addSymbol(0x1D51F, "mfrakb", "mathalpha", "mathematical fraktur small b")
addSymbol(0x1D520, "mfrakc", "mathalpha", "mathematical fraktur small c")
addSymbol(0x1D521, "mfrakd", "mathalpha", "mathematical fraktur small d")
addSymbol(0x1D522, "mfrake", "mathalpha", "mathematical fraktur small e")
addSymbol(0x1D523, "mfrakf", "mathalpha", "mathematical fraktur small f")
addSymbol(0x1D524, "mfrakg", "mathalpha", "mathematical fraktur small g")
addSymbol(0x1D525, "mfrakh", "mathalpha", "mathematical fraktur small h")
addSymbol(0x1D526, "mfraki", "mathalpha", "mathematical fraktur small i")
addSymbol(0x1D527, "mfrakj", "mathalpha", "mathematical fraktur small j")
addSymbol(0x1D528, "mfrakk", "mathalpha", "mathematical fraktur small k")
addSymbol(0x1D529, "mfrakl", "mathalpha", "mathematical fraktur small l")
addSymbol(0x1D52A, "mfrakm", "mathalpha", "mathematical fraktur small m")
addSymbol(0x1D52B, "mfrakn", "mathalpha", "mathematical fraktur small n")
addSymbol(0x1D52C, "mfrako", "mathalpha", "mathematical fraktur small o")
addSymbol(0x1D52D, "mfrakp", "mathalpha", "mathematical fraktur small p")
addSymbol(0x1D52E, "mfrakq", "mathalpha", "mathematical fraktur small q")
addSymbol(0x1D52F, "mfrakr", "mathalpha", "mathematical fraktur small r")
addSymbol(0x1D530, "mfraks", "mathalpha", "mathematical fraktur small s")
addSymbol(0x1D531, "mfrakt", "mathalpha", "mathematical fraktur small t")
addSymbol(0x1D532, "mfraku", "mathalpha", "mathematical fraktur small u")
addSymbol(0x1D533, "mfrakv", "mathalpha", "mathematical fraktur small v")
addSymbol(0x1D534, "mfrakw", "mathalpha", "mathematical fraktur small w")
addSymbol(0x1D535, "mfrakx", "mathalpha", "mathematical fraktur small x")
addSymbol(0x1D536, "mfraky", "mathalpha", "mathematical fraktur small y")
addSymbol(0x1D537, "mfrakz", "mathalpha", "mathematical fraktur small z")
addSymbol(0x1D538, "BbbA", "mathalpha", "mathematical double-struck capital a")
addSymbol(0x1D539, "BbbB", "mathalpha", "mathematical double-struck capital b")
addSymbol(0x1D53B, "BbbD", "mathalpha", "mathematical double-struck capital d")
addSymbol(0x1D53C, "BbbE", "mathalpha", "mathematical double-struck capital e")
addSymbol(0x1D53D, "BbbF", "mathalpha", "mathematical double-struck capital f")
addSymbol(0x1D53E, "BbbG", "mathalpha", "mathematical double-struck capital g")
addSymbol(0x1D540, "BbbI", "mathalpha", "mathematical double-struck capital i")
addSymbol(0x1D541, "BbbJ", "mathalpha", "mathematical double-struck capital j")
addSymbol(0x1D542, "BbbK", "mathalpha", "mathematical double-struck capital k")
addSymbol(0x1D543, "BbbL", "mathalpha", "mathematical double-struck capital l")
addSymbol(0x1D544, "BbbM", "mathalpha", "mathematical double-struck capital m")
addSymbol(0x1D546, "BbbO", "mathalpha", "mathematical double-struck capital o")
addSymbol(0x1D54A, "BbbS", "mathalpha", "mathematical double-struck capital s")
addSymbol(0x1D54B, "BbbT", "mathalpha", "mathematical double-struck capital t")
addSymbol(0x1D54C, "BbbU", "mathalpha", "mathematical double-struck capital u")
addSymbol(0x1D54D, "BbbV", "mathalpha", "mathematical double-struck capital v")
addSymbol(0x1D54E, "BbbW", "mathalpha", "mathematical double-struck capital w")
addSymbol(0x1D54F, "BbbX", "mathalpha", "mathematical double-struck capital x")
addSymbol(0x1D550, "BbbY", "mathalpha", "mathematical double-struck capital y")
addSymbol(0x1D552, "Bbba", "mathalpha", "mathematical double-struck small a")
addSymbol(0x1D553, "Bbbb", "mathalpha", "mathematical double-struck small b")
addSymbol(0x1D554, "Bbbc", "mathalpha", "mathematical double-struck small c")
addSymbol(0x1D555, "Bbbd", "mathalpha", "mathematical double-struck small d")
addSymbol(0x1D556, "Bbbe", "mathalpha", "mathematical double-struck small e")
addSymbol(0x1D557, "Bbbf", "mathalpha", "mathematical double-struck small f")
addSymbol(0x1D558, "Bbbg", "mathalpha", "mathematical double-struck small g")
addSymbol(0x1D559, "Bbbh", "mathalpha", "mathematical double-struck small h")
addSymbol(0x1D55A, "Bbbi", "mathalpha", "mathematical double-struck small i")
addSymbol(0x1D55B, "Bbbj", "mathalpha", "mathematical double-struck small j")
addSymbol(0x1D55C, "Bbbk", "mathalpha", "mathematical double-struck small k")
addSymbol(0x1D55D, "Bbbl", "mathalpha", "mathematical double-struck small l")
addSymbol(0x1D55E, "Bbbm", "mathalpha", "mathematical double-struck small m")
addSymbol(0x1D55F, "Bbbn", "mathalpha", "mathematical double-struck small n")
addSymbol(0x1D560, "Bbbo", "mathalpha", "mathematical double-struck small o")
addSymbol(0x1D561, "Bbbp", "mathalpha", "mathematical double-struck small p")
addSymbol(0x1D562, "Bbbq", "mathalpha", "mathematical double-struck small q")
addSymbol(0x1D563, "Bbbr", "mathalpha", "mathematical double-struck small r")
addSymbol(0x1D564, "Bbbs", "mathalpha", "mathematical double-struck small s")
addSymbol(0x1D565, "Bbbt", "mathalpha", "mathematical double-struck small t")
addSymbol(0x1D566, "Bbbu", "mathalpha", "mathematical double-struck small u")
addSymbol(0x1D567, "Bbbv", "mathalpha", "mathematical double-struck small v")
addSymbol(0x1D568, "Bbbw", "mathalpha", "mathematical double-struck small w")
addSymbol(0x1D569, "Bbbx", "mathalpha", "mathematical double-struck small x")
addSymbol(0x1D56A, "Bbby", "mathalpha", "mathematical double-struck small y")
addSymbol(0x1D56B, "Bbbz", "mathalpha", "mathematical double-struck small z")
addSymbol(0x1D56C, "mbffrakA", "mathalpha", "mathematical bold fraktur capital a")
addSymbol(0x1D56D, "mbffrakB", "mathalpha", "mathematical bold fraktur capital b")
addSymbol(0x1D56E, "mbffrakC", "mathalpha", "mathematical bold fraktur capital c")
addSymbol(0x1D56F, "mbffrakD", "mathalpha", "mathematical bold fraktur capital d")
addSymbol(0x1D570, "mbffrakE", "mathalpha", "mathematical bold fraktur capital e")
addSymbol(0x1D571, "mbffrakF", "mathalpha", "mathematical bold fraktur capital f")
addSymbol(0x1D572, "mbffrakG", "mathalpha", "mathematical bold fraktur capital g")
addSymbol(0x1D573, "mbffrakH", "mathalpha", "mathematical bold fraktur capital h")
addSymbol(0x1D574, "mbffrakI", "mathalpha", "mathematical bold fraktur capital i")
addSymbol(0x1D575, "mbffrakJ", "mathalpha", "mathematical bold fraktur capital j")
addSymbol(0x1D576, "mbffrakK", "mathalpha", "mathematical bold fraktur capital k")
addSymbol(0x1D577, "mbffrakL", "mathalpha", "mathematical bold fraktur capital l")
addSymbol(0x1D578, "mbffrakM", "mathalpha", "mathematical bold fraktur capital m")
addSymbol(0x1D579, "mbffrakN", "mathalpha", "mathematical bold fraktur capital n")
addSymbol(0x1D57A, "mbffrakO", "mathalpha", "mathematical bold fraktur capital o")
addSymbol(0x1D57B, "mbffrakP", "mathalpha", "mathematical bold fraktur capital p")
addSymbol(0x1D57C, "mbffrakQ", "mathalpha", "mathematical bold fraktur capital q")
addSymbol(0x1D57D, "mbffrakR", "mathalpha", "mathematical bold fraktur capital r")
addSymbol(0x1D57E, "mbffrakS", "mathalpha", "mathematical bold fraktur capital s")
addSymbol(0x1D57F, "mbffrakT", "mathalpha", "mathematical bold fraktur capital t")
addSymbol(0x1D580, "mbffrakU", "mathalpha", "mathematical bold fraktur capital u")
addSymbol(0x1D581, "mbffrakV", "mathalpha", "mathematical bold fraktur capital v")
addSymbol(0x1D582, "mbffrakW", "mathalpha", "mathematical bold fraktur capital w")
addSymbol(0x1D583, "mbffrakX", "mathalpha", "mathematical bold fraktur capital x")
addSymbol(0x1D584, "mbffrakY", "mathalpha", "mathematical bold fraktur capital y")
addSymbol(0x1D585, "mbffrakZ", "mathalpha", "mathematical bold fraktur capital z")
addSymbol(0x1D586, "mbffraka", "mathalpha", "mathematical bold fraktur small a")
addSymbol(0x1D587, "mbffrakb", "mathalpha", "mathematical bold fraktur small b")
addSymbol(0x1D588, "mbffrakc", "mathalpha", "mathematical bold fraktur small c")
addSymbol(0x1D589, "mbffrakd", "mathalpha", "mathematical bold fraktur small d")
addSymbol(0x1D58A, "mbffrake", "mathalpha", "mathematical bold fraktur small e")
addSymbol(0x1D58B, "mbffrakf", "mathalpha", "mathematical bold fraktur small f")
addSymbol(0x1D58C, "mbffrakg", "mathalpha", "mathematical bold fraktur small g")
addSymbol(0x1D58D, "mbffrakh", "mathalpha", "mathematical bold fraktur small h")
addSymbol(0x1D58E, "mbffraki", "mathalpha", "mathematical bold fraktur small i")
addSymbol(0x1D58F, "mbffrakj", "mathalpha", "mathematical bold fraktur small j")
addSymbol(0x1D590, "mbffrakk", "mathalpha", "mathematical bold fraktur small k")
addSymbol(0x1D591, "mbffrakl", "mathalpha", "mathematical bold fraktur small l")
addSymbol(0x1D592, "mbffrakm", "mathalpha", "mathematical bold fraktur small m")
addSymbol(0x1D593, "mbffrakn", "mathalpha", "mathematical bold fraktur small n")
addSymbol(0x1D594, "mbffrako", "mathalpha", "mathematical bold fraktur small o")
addSymbol(0x1D595, "mbffrakp", "mathalpha", "mathematical bold fraktur small p")
addSymbol(0x1D596, "mbffrakq", "mathalpha", "mathematical bold fraktur small q")
addSymbol(0x1D597, "mbffrakr", "mathalpha", "mathematical bold fraktur small r")
addSymbol(0x1D598, "mbffraks", "mathalpha", "mathematical bold fraktur small s")
addSymbol(0x1D599, "mbffrakt", "mathalpha", "mathematical bold fraktur small t")
addSymbol(0x1D59A, "mbffraku", "mathalpha", "mathematical bold fraktur small u")
addSymbol(0x1D59B, "mbffrakv", "mathalpha", "mathematical bold fraktur small v")
addSymbol(0x1D59C, "mbffrakw", "mathalpha", "mathematical bold fraktur small w")
addSymbol(0x1D59D, "mbffrakx", "mathalpha", "mathematical bold fraktur small x")
addSymbol(0x1D59E, "mbffraky", "mathalpha", "mathematical bold fraktur small y")
addSymbol(0x1D59F, "mbffrakz", "mathalpha", "mathematical bold fraktur small z")
addSymbol(0x1D5A0, "msansA", "mathalpha", "mathematical sans-serif capital a")
addSymbol(0x1D5A1, "msansB", "mathalpha", "mathematical sans-serif capital b")
addSymbol(0x1D5A2, "msansC", "mathalpha", "mathematical sans-serif capital c")
addSymbol(0x1D5A3, "msansD", "mathalpha", "mathematical sans-serif capital d")
addSymbol(0x1D5A4, "msansE", "mathalpha", "mathematical sans-serif capital e")
addSymbol(0x1D5A5, "msansF", "mathalpha", "mathematical sans-serif capital f")
addSymbol(0x1D5A6, "msansG", "mathalpha", "mathematical sans-serif capital g")
addSymbol(0x1D5A7, "msansH", "mathalpha", "mathematical sans-serif capital h")
addSymbol(0x1D5A8, "msansI", "mathalpha", "mathematical sans-serif capital i")
addSymbol(0x1D5A9, "msansJ", "mathalpha", "mathematical sans-serif capital j")
addSymbol(0x1D5AA, "msansK", "mathalpha", "mathematical sans-serif capital k")
addSymbol(0x1D5AB, "msansL", "mathalpha", "mathematical sans-serif capital l")
addSymbol(0x1D5AC, "msansM", "mathalpha", "mathematical sans-serif capital m")
addSymbol(0x1D5AD, "msansN", "mathalpha", "mathematical sans-serif capital n")
addSymbol(0x1D5AE, "msansO", "mathalpha", "mathematical sans-serif capital o")
addSymbol(0x1D5AF, "msansP", "mathalpha", "mathematical sans-serif capital p")
addSymbol(0x1D5B0, "msansQ", "mathalpha", "mathematical sans-serif capital q")
addSymbol(0x1D5B1, "msansR", "mathalpha", "mathematical sans-serif capital r")
addSymbol(0x1D5B2, "msansS", "mathalpha", "mathematical sans-serif capital s")
addSymbol(0x1D5B3, "msansT", "mathalpha", "mathematical sans-serif capital t")
addSymbol(0x1D5B4, "msansU", "mathalpha", "mathematical sans-serif capital u")
addSymbol(0x1D5B5, "msansV", "mathalpha", "mathematical sans-serif capital v")
addSymbol(0x1D5B6, "msansW", "mathalpha", "mathematical sans-serif capital w")
addSymbol(0x1D5B7, "msansX", "mathalpha", "mathematical sans-serif capital x")
addSymbol(0x1D5B8, "msansY", "mathalpha", "mathematical sans-serif capital y")
addSymbol(0x1D5B9, "msansZ", "mathalpha", "mathematical sans-serif capital z")
addSymbol(0x1D5BA, "msansa", "mathalpha", "mathematical sans-serif small a")
addSymbol(0x1D5BB, "msansb", "mathalpha", "mathematical sans-serif small b")
addSymbol(0x1D5BC, "msansc", "mathalpha", "mathematical sans-serif small c")
addSymbol(0x1D5BD, "msansd", "mathalpha", "mathematical sans-serif small d")
addSymbol(0x1D5BE, "msanse", "mathalpha", "mathematical sans-serif small e")
addSymbol(0x1D5BF, "msansf", "mathalpha", "mathematical sans-serif small f")
addSymbol(0x1D5C0, "msansg", "mathalpha", "mathematical sans-serif small g")
addSymbol(0x1D5C1, "msansh", "mathalpha", "mathematical sans-serif small h")
addSymbol(0x1D5C2, "msansi", "mathalpha", "mathematical sans-serif small i")
addSymbol(0x1D5C3, "msansj", "mathalpha", "mathematical sans-serif small j")
addSymbol(0x1D5C4, "msansk", "mathalpha", "mathematical sans-serif small k")
addSymbol(0x1D5C5, "msansl", "mathalpha", "mathematical sans-serif small l")
addSymbol(0x1D5C6, "msansm", "mathalpha", "mathematical sans-serif small m")
addSymbol(0x1D5C7, "msansn", "mathalpha", "mathematical sans-serif small n")
addSymbol(0x1D5C8, "msanso", "mathalpha", "mathematical sans-serif small o")
addSymbol(0x1D5C9, "msansp", "mathalpha", "mathematical sans-serif small p")
addSymbol(0x1D5CA, "msansq", "mathalpha", "mathematical sans-serif small q")
addSymbol(0x1D5CB, "msansr", "mathalpha", "mathematical sans-serif small r")
addSymbol(0x1D5CC, "msanss", "mathalpha", "mathematical sans-serif small s")
addSymbol(0x1D5CD, "msanst", "mathalpha", "mathematical sans-serif small t")
addSymbol(0x1D5CE, "msansu", "mathalpha", "mathematical sans-serif small u")
addSymbol(0x1D5CF, "msansv", "mathalpha", "mathematical sans-serif small v")
addSymbol(0x1D5D0, "msansw", "mathalpha", "mathematical sans-serif small w")
addSymbol(0x1D5D1, "msansx", "mathalpha", "mathematical sans-serif small x")
addSymbol(0x1D5D2, "msansy", "mathalpha", "mathematical sans-serif small y")
addSymbol(0x1D5D3, "msansz", "mathalpha", "mathematical sans-serif small z")
addSymbol(0x1D5D4, "mbfsansA", "mathalpha", "mathematical sans-serif bold capital a")
addSymbol(0x1D5D5, "mbfsansB", "mathalpha", "mathematical sans-serif bold capital b")
addSymbol(0x1D5D6, "mbfsansC", "mathalpha", "mathematical sans-serif bold capital c")
addSymbol(0x1D5D7, "mbfsansD", "mathalpha", "mathematical sans-serif bold capital d")
addSymbol(0x1D5D8, "mbfsansE", "mathalpha", "mathematical sans-serif bold capital e")
addSymbol(0x1D5D9, "mbfsansF", "mathalpha", "mathematical sans-serif bold capital f")
addSymbol(0x1D5DA, "mbfsansG", "mathalpha", "mathematical sans-serif bold capital g")
addSymbol(0x1D5DB, "mbfsansH", "mathalpha", "mathematical sans-serif bold capital h")
addSymbol(0x1D5DC, "mbfsansI", "mathalpha", "mathematical sans-serif bold capital i")
addSymbol(0x1D5DD, "mbfsansJ", "mathalpha", "mathematical sans-serif bold capital j")
addSymbol(0x1D5DE, "mbfsansK", "mathalpha", "mathematical sans-serif bold capital k")
addSymbol(0x1D5DF, "mbfsansL", "mathalpha", "mathematical sans-serif bold capital l")
addSymbol(0x1D5E0, "mbfsansM", "mathalpha", "mathematical sans-serif bold capital m")
addSymbol(0x1D5E1, "mbfsansN", "mathalpha", "mathematical sans-serif bold capital n")
addSymbol(0x1D5E2, "mbfsansO", "mathalpha", "mathematical sans-serif bold capital o")
addSymbol(0x1D5E3, "mbfsansP", "mathalpha", "mathematical sans-serif bold capital p")
addSymbol(0x1D5E4, "mbfsansQ", "mathalpha", "mathematical sans-serif bold capital q")
addSymbol(0x1D5E5, "mbfsansR", "mathalpha", "mathematical sans-serif bold capital r")
addSymbol(0x1D5E6, "mbfsansS", "mathalpha", "mathematical sans-serif bold capital s")
addSymbol(0x1D5E7, "mbfsansT", "mathalpha", "mathematical sans-serif bold capital t")
addSymbol(0x1D5E8, "mbfsansU", "mathalpha", "mathematical sans-serif bold capital u")
addSymbol(0x1D5E9, "mbfsansV", "mathalpha", "mathematical sans-serif bold capital v")
addSymbol(0x1D5EA, "mbfsansW", "mathalpha", "mathematical sans-serif bold capital w")
addSymbol(0x1D5EB, "mbfsansX", "mathalpha", "mathematical sans-serif bold capital x")
addSymbol(0x1D5EC, "mbfsansY", "mathalpha", "mathematical sans-serif bold capital y")
addSymbol(0x1D5ED, "mbfsansZ", "mathalpha", "mathematical sans-serif bold capital z")
addSymbol(0x1D5EE, "mbfsansa", "mathalpha", "mathematical sans-serif bold small a")
addSymbol(0x1D5EF, "mbfsansb", "mathalpha", "mathematical sans-serif bold small b")
addSymbol(0x1D5F0, "mbfsansc", "mathalpha", "mathematical sans-serif bold small c")
addSymbol(0x1D5F1, "mbfsansd", "mathalpha", "mathematical sans-serif bold small d")
addSymbol(0x1D5F2, "mbfsanse", "mathalpha", "mathematical sans-serif bold small e")
addSymbol(0x1D5F3, "mbfsansf", "mathalpha", "mathematical sans-serif bold small f")
addSymbol(0x1D5F4, "mbfsansg", "mathalpha", "mathematical sans-serif bold small g")
addSymbol(0x1D5F5, "mbfsansh", "mathalpha", "mathematical sans-serif bold small h")
addSymbol(0x1D5F6, "mbfsansi", "mathalpha", "mathematical sans-serif bold small i")
addSymbol(0x1D5F7, "mbfsansj", "mathalpha", "mathematical sans-serif bold small j")
addSymbol(0x1D5F8, "mbfsansk", "mathalpha", "mathematical sans-serif bold small k")
addSymbol(0x1D5F9, "mbfsansl", "mathalpha", "mathematical sans-serif bold small l")
addSymbol(0x1D5FA, "mbfsansm", "mathalpha", "mathematical sans-serif bold small m")
addSymbol(0x1D5FB, "mbfsansn", "mathalpha", "mathematical sans-serif bold small n")
addSymbol(0x1D5FC, "mbfsanso", "mathalpha", "mathematical sans-serif bold small o")
addSymbol(0x1D5FD, "mbfsansp", "mathalpha", "mathematical sans-serif bold small p")
addSymbol(0x1D5FE, "mbfsansq", "mathalpha", "mathematical sans-serif bold small q")
addSymbol(0x1D5FF, "mbfsansr", "mathalpha", "mathematical sans-serif bold small r")
addSymbol(0x1D600, "mbfsanss", "mathalpha", "mathematical sans-serif bold small s")
addSymbol(0x1D601, "mbfsanst", "mathalpha", "mathematical sans-serif bold small t")
addSymbol(0x1D602, "mbfsansu", "mathalpha", "mathematical sans-serif bold small u")
addSymbol(0x1D603, "mbfsansv", "mathalpha", "mathematical sans-serif bold small v")
addSymbol(0x1D604, "mbfsansw", "mathalpha", "mathematical sans-serif bold small w")
addSymbol(0x1D605, "mbfsansx", "mathalpha", "mathematical sans-serif bold small x")
addSymbol(0x1D606, "mbfsansy", "mathalpha", "mathematical sans-serif bold small y")
addSymbol(0x1D607, "mbfsansz", "mathalpha", "mathematical sans-serif bold small z")
addSymbol(0x1D608, "mitsansA", "mathalpha", "mathematical sans-serif italic capital a")
addSymbol(0x1D609, "mitsansB", "mathalpha", "mathematical sans-serif italic capital b")
addSymbol(0x1D60A, "mitsansC", "mathalpha", "mathematical sans-serif italic capital c")
addSymbol(0x1D60B, "mitsansD", "mathalpha", "mathematical sans-serif italic capital d")
addSymbol(0x1D60C, "mitsansE", "mathalpha", "mathematical sans-serif italic capital e")
addSymbol(0x1D60D, "mitsansF", "mathalpha", "mathematical sans-serif italic capital f")
addSymbol(0x1D60E, "mitsansG", "mathalpha", "mathematical sans-serif italic capital g")
addSymbol(0x1D60F, "mitsansH", "mathalpha", "mathematical sans-serif italic capital h")
addSymbol(0x1D610, "mitsansI", "mathalpha", "mathematical sans-serif italic capital i")
addSymbol(0x1D611, "mitsansJ", "mathalpha", "mathematical sans-serif italic capital j")
addSymbol(0x1D612, "mitsansK", "mathalpha", "mathematical sans-serif italic capital k")
addSymbol(0x1D613, "mitsansL", "mathalpha", "mathematical sans-serif italic capital l")
addSymbol(0x1D614, "mitsansM", "mathalpha", "mathematical sans-serif italic capital m")
addSymbol(0x1D615, "mitsansN", "mathalpha", "mathematical sans-serif italic capital n")
addSymbol(0x1D616, "mitsansO", "mathalpha", "mathematical sans-serif italic capital o")
addSymbol(0x1D617, "mitsansP", "mathalpha", "mathematical sans-serif italic capital p")
addSymbol(0x1D618, "mitsansQ", "mathalpha", "mathematical sans-serif italic capital q")
addSymbol(0x1D619, "mitsansR", "mathalpha", "mathematical sans-serif italic capital r")
addSymbol(0x1D61A, "mitsansS", "mathalpha", "mathematical sans-serif italic capital s")
addSymbol(0x1D61B, "mitsansT", "mathalpha", "mathematical sans-serif italic capital t")
addSymbol(0x1D61C, "mitsansU", "mathalpha", "mathematical sans-serif italic capital u")
addSymbol(0x1D61D, "mitsansV", "mathalpha", "mathematical sans-serif italic capital v")
addSymbol(0x1D61E, "mitsansW", "mathalpha", "mathematical sans-serif italic capital w")
addSymbol(0x1D61F, "mitsansX", "mathalpha", "mathematical sans-serif italic capital x")
addSymbol(0x1D620, "mitsansY", "mathalpha", "mathematical sans-serif italic capital y")
addSymbol(0x1D621, "mitsansZ", "mathalpha", "mathematical sans-serif italic capital z")
addSymbol(0x1D622, "mitsansa", "mathalpha", "mathematical sans-serif italic small a")
addSymbol(0x1D623, "mitsansb", "mathalpha", "mathematical sans-serif italic small b")
addSymbol(0x1D624, "mitsansc", "mathalpha", "mathematical sans-serif italic small c")
addSymbol(0x1D625, "mitsansd", "mathalpha", "mathematical sans-serif italic small d")
addSymbol(0x1D626, "mitsanse", "mathalpha", "mathematical sans-serif italic small e")
addSymbol(0x1D627, "mitsansf", "mathalpha", "mathematical sans-serif italic small f")
addSymbol(0x1D628, "mitsansg", "mathalpha", "mathematical sans-serif italic small g")
addSymbol(0x1D629, "mitsansh", "mathalpha", "mathematical sans-serif italic small h")
addSymbol(0x1D62A, "mitsansi", "mathalpha", "mathematical sans-serif italic small i")
addSymbol(0x1D62B, "mitsansj", "mathalpha", "mathematical sans-serif italic small j")
addSymbol(0x1D62C, "mitsansk", "mathalpha", "mathematical sans-serif italic small k")
addSymbol(0x1D62D, "mitsansl", "mathalpha", "mathematical sans-serif italic small l")
addSymbol(0x1D62E, "mitsansm", "mathalpha", "mathematical sans-serif italic small m")
addSymbol(0x1D62F, "mitsansn", "mathalpha", "mathematical sans-serif italic small n")
addSymbol(0x1D630, "mitsanso", "mathalpha", "mathematical sans-serif italic small o")
addSymbol(0x1D631, "mitsansp", "mathalpha", "mathematical sans-serif italic small p")
addSymbol(0x1D632, "mitsansq", "mathalpha", "mathematical sans-serif italic small q")
addSymbol(0x1D633, "mitsansr", "mathalpha", "mathematical sans-serif italic small r")
addSymbol(0x1D634, "mitsanss", "mathalpha", "mathematical sans-serif italic small s")
addSymbol(0x1D635, "mitsanst", "mathalpha", "mathematical sans-serif italic small t")
addSymbol(0x1D636, "mitsansu", "mathalpha", "mathematical sans-serif italic small u")
addSymbol(0x1D637, "mitsansv", "mathalpha", "mathematical sans-serif italic small v")
addSymbol(0x1D638, "mitsansw", "mathalpha", "mathematical sans-serif italic small w")
addSymbol(0x1D639, "mitsansx", "mathalpha", "mathematical sans-serif italic small x")
addSymbol(0x1D63A, "mitsansy", "mathalpha", "mathematical sans-serif italic small y")
addSymbol(0x1D63B, "mitsansz", "mathalpha", "mathematical sans-serif italic small z")
addSymbol(0x1D63C, "mbfitsansA", "mathalpha", "mathematical sans-serif bold italic capital a")
addSymbol(0x1D63D, "mbfitsansB", "mathalpha", "mathematical sans-serif bold italic capital b")
addSymbol(0x1D63E, "mbfitsansC", "mathalpha", "mathematical sans-serif bold italic capital c")
addSymbol(0x1D63F, "mbfitsansD", "mathalpha", "mathematical sans-serif bold italic capital d")
addSymbol(0x1D640, "mbfitsansE", "mathalpha", "mathematical sans-serif bold italic capital e")
addSymbol(0x1D641, "mbfitsansF", "mathalpha", "mathematical sans-serif bold italic capital f")
addSymbol(0x1D642, "mbfitsansG", "mathalpha", "mathematical sans-serif bold italic capital g")
addSymbol(0x1D643, "mbfitsansH", "mathalpha", "mathematical sans-serif bold italic capital h")
addSymbol(0x1D644, "mbfitsansI", "mathalpha", "mathematical sans-serif bold italic capital i")
addSymbol(0x1D645, "mbfitsansJ", "mathalpha", "mathematical sans-serif bold italic capital j")
addSymbol(0x1D646, "mbfitsansK", "mathalpha", "mathematical sans-serif bold italic capital k")
addSymbol(0x1D647, "mbfitsansL", "mathalpha", "mathematical sans-serif bold italic capital l")
addSymbol(0x1D648, "mbfitsansM", "mathalpha", "mathematical sans-serif bold italic capital m")
addSymbol(0x1D649, "mbfitsansN", "mathalpha", "mathematical sans-serif bold italic capital n")
addSymbol(0x1D64A, "mbfitsansO", "mathalpha", "mathematical sans-serif bold italic capital o")
addSymbol(0x1D64B, "mbfitsansP", "mathalpha", "mathematical sans-serif bold italic capital p")
addSymbol(0x1D64C, "mbfitsansQ", "mathalpha", "mathematical sans-serif bold italic capital q")
addSymbol(0x1D64D, "mbfitsansR", "mathalpha", "mathematical sans-serif bold italic capital r")
addSymbol(0x1D64E, "mbfitsansS", "mathalpha", "mathematical sans-serif bold italic capital s")
addSymbol(0x1D64F, "mbfitsansT", "mathalpha", "mathematical sans-serif bold italic capital t")
addSymbol(0x1D650, "mbfitsansU", "mathalpha", "mathematical sans-serif bold italic capital u")
addSymbol(0x1D651, "mbfitsansV", "mathalpha", "mathematical sans-serif bold italic capital v")
addSymbol(0x1D652, "mbfitsansW", "mathalpha", "mathematical sans-serif bold italic capital w")
addSymbol(0x1D653, "mbfitsansX", "mathalpha", "mathematical sans-serif bold italic capital x")
addSymbol(0x1D654, "mbfitsansY", "mathalpha", "mathematical sans-serif bold italic capital y")
addSymbol(0x1D655, "mbfitsansZ", "mathalpha", "mathematical sans-serif bold italic capital z")
addSymbol(0x1D656, "mbfitsansa", "mathalpha", "mathematical sans-serif bold italic small a")
addSymbol(0x1D657, "mbfitsansb", "mathalpha", "mathematical sans-serif bold italic small b")
addSymbol(0x1D658, "mbfitsansc", "mathalpha", "mathematical sans-serif bold italic small c")
addSymbol(0x1D659, "mbfitsansd", "mathalpha", "mathematical sans-serif bold italic small d")
addSymbol(0x1D65A, "mbfitsanse", "mathalpha", "mathematical sans-serif bold italic small e")
addSymbol(0x1D65B, "mbfitsansf", "mathalpha", "mathematical sans-serif bold italic small f")
addSymbol(0x1D65C, "mbfitsansg", "mathalpha", "mathematical sans-serif bold italic small g")
addSymbol(0x1D65D, "mbfitsansh", "mathalpha", "mathematical sans-serif bold italic small h")
addSymbol(0x1D65E, "mbfitsansi", "mathalpha", "mathematical sans-serif bold italic small i")
addSymbol(0x1D65F, "mbfitsansj", "mathalpha", "mathematical sans-serif bold italic small j")
addSymbol(0x1D660, "mbfitsansk", "mathalpha", "mathematical sans-serif bold italic small k")
addSymbol(0x1D661, "mbfitsansl", "mathalpha", "mathematical sans-serif bold italic small l")
addSymbol(0x1D662, "mbfitsansm", "mathalpha", "mathematical sans-serif bold italic small m")
addSymbol(0x1D663, "mbfitsansn", "mathalpha", "mathematical sans-serif bold italic small n")
addSymbol(0x1D664, "mbfitsanso", "mathalpha", "mathematical sans-serif bold italic small o")
addSymbol(0x1D665, "mbfitsansp", "mathalpha", "mathematical sans-serif bold italic small p")
addSymbol(0x1D666, "mbfitsansq", "mathalpha", "mathematical sans-serif bold italic small q")
addSymbol(0x1D667, "mbfitsansr", "mathalpha", "mathematical sans-serif bold italic small r")
addSymbol(0x1D668, "mbfitsanss", "mathalpha", "mathematical sans-serif bold italic small s")
addSymbol(0x1D669, "mbfitsanst", "mathalpha", "mathematical sans-serif bold italic small t")
addSymbol(0x1D66A, "mbfitsansu", "mathalpha", "mathematical sans-serif bold italic small u")
addSymbol(0x1D66B, "mbfitsansv", "mathalpha", "mathematical sans-serif bold italic small v")
addSymbol(0x1D66C, "mbfitsansw", "mathalpha", "mathematical sans-serif bold italic small w")
addSymbol(0x1D66D, "mbfitsansx", "mathalpha", "mathematical sans-serif bold italic small x")
addSymbol(0x1D66E, "mbfitsansy", "mathalpha", "mathematical sans-serif bold italic small y")
addSymbol(0x1D66F, "mbfitsansz", "mathalpha", "mathematical sans-serif bold italic small z")
addSymbol(0x1D670, "mttA", "mathalpha", "mathematical monospace capital a")
addSymbol(0x1D671, "mttB", "mathalpha", "mathematical monospace capital b")
addSymbol(0x1D672, "mttC", "mathalpha", "mathematical monospace capital c")
addSymbol(0x1D673, "mttD", "mathalpha", "mathematical monospace capital d")
addSymbol(0x1D674, "mttE", "mathalpha", "mathematical monospace capital e")
addSymbol(0x1D675, "mttF", "mathalpha", "mathematical monospace capital f")
addSymbol(0x1D676, "mttG", "mathalpha", "mathematical monospace capital g")
addSymbol(0x1D677, "mttH", "mathalpha", "mathematical monospace capital h")
addSymbol(0x1D678, "mttI", "mathalpha", "mathematical monospace capital i")
addSymbol(0x1D679, "mttJ", "mathalpha", "mathematical monospace capital j")
addSymbol(0x1D67A, "mttK", "mathalpha", "mathematical monospace capital k")
addSymbol(0x1D67B, "mttL", "mathalpha", "mathematical monospace capital l")
addSymbol(0x1D67C, "mttM", "mathalpha", "mathematical monospace capital m")
addSymbol(0x1D67D, "mttN", "mathalpha", "mathematical monospace capital n")
addSymbol(0x1D67E, "mttO", "mathalpha", "mathematical monospace capital o")
addSymbol(0x1D67F, "mttP", "mathalpha", "mathematical monospace capital p")
addSymbol(0x1D680, "mttQ", "mathalpha", "mathematical monospace capital q")
addSymbol(0x1D681, "mttR", "mathalpha", "mathematical monospace capital r")
addSymbol(0x1D682, "mttS", "mathalpha", "mathematical monospace capital s")
addSymbol(0x1D683, "mttT", "mathalpha", "mathematical monospace capital t")
addSymbol(0x1D684, "mttU", "mathalpha", "mathematical monospace capital u")
addSymbol(0x1D685, "mttV", "mathalpha", "mathematical monospace capital v")
addSymbol(0x1D686, "mttW", "mathalpha", "mathematical monospace capital w")
addSymbol(0x1D687, "mttX", "mathalpha", "mathematical monospace capital x")
addSymbol(0x1D688, "mttY", "mathalpha", "mathematical monospace capital y")
addSymbol(0x1D689, "mttZ", "mathalpha", "mathematical monospace capital z")
addSymbol(0x1D68A, "mtta", "mathalpha", "mathematical monospace small a")
addSymbol(0x1D68B, "mttb", "mathalpha", "mathematical monospace small b")
addSymbol(0x1D68C, "mttc", "mathalpha", "mathematical monospace small c")
addSymbol(0x1D68D, "mttd", "mathalpha", "mathematical monospace small d")
addSymbol(0x1D68E, "mtte", "mathalpha", "mathematical monospace small e")
addSymbol(0x1D68F, "mttf", "mathalpha", "mathematical monospace small f")
addSymbol(0x1D690, "mttg", "mathalpha", "mathematical monospace small g")
addSymbol(0x1D691, "mtth", "mathalpha", "mathematical monospace small h")
addSymbol(0x1D692, "mtti", "mathalpha", "mathematical monospace small i")
addSymbol(0x1D693, "mttj", "mathalpha", "mathematical monospace small j")
addSymbol(0x1D694, "mttk", "mathalpha", "mathematical monospace small k")
addSymbol(0x1D695, "mttl", "mathalpha", "mathematical monospace small l")
addSymbol(0x1D696, "mttm", "mathalpha", "mathematical monospace small m")
addSymbol(0x1D697, "mttn", "mathalpha", "mathematical monospace small n")
addSymbol(0x1D698, "mtto", "mathalpha", "mathematical monospace small o")
addSymbol(0x1D699, "mttp", "mathalpha", "mathematical monospace small p")
addSymbol(0x1D69A, "mttq", "mathalpha", "mathematical monospace small q")
addSymbol(0x1D69B, "mttr", "mathalpha", "mathematical monospace small r")
addSymbol(0x1D69C, "mtts", "mathalpha", "mathematical monospace small s")
addSymbol(0x1D69D, "mttt", "mathalpha", "mathematical monospace small t")
addSymbol(0x1D69E, "mttu", "mathalpha", "mathematical monospace small u")
addSymbol(0x1D69F, "mttv", "mathalpha", "mathematical monospace small v")
addSymbol(0x1D6A0, "mttw", "mathalpha", "mathematical monospace small w")
addSymbol(0x1D6A1, "mttx", "mathalpha", "mathematical monospace small x")
addSymbol(0x1D6A2, "mtty", "mathalpha", "mathematical monospace small y")
addSymbol(0x1D6A3, "mttz", "mathalpha", "mathematical monospace small z")
addSymbol(0x1D6A4, "imath", "mathalpha", "mathematical italic small dotless i")
addSymbol(0x1D6A5, "jmath", "mathalpha", "mathematical italic small dotless j")
addSymbol(0x1D6A8, "mbfAlpha", "mathalpha", "mathematical bold capital alpha")
addSymbol(0x1D6A9, "mbfBeta", "mathalpha", "mathematical bold capital beta")
addSymbol(0x1D6AA, "mbfGamma", "mathalpha", "mathematical bold capital gamma")
addSymbol(0x1D6AB, "mbfDelta", "mathalpha", "mathematical bold capital delta")
addSymbol(0x1D6AC, "mbfEpsilon", "mathalpha", "mathematical bold capital epsilon")
addSymbol(0x1D6AD, "mbfZeta", "mathalpha", "mathematical bold capital zeta")
addSymbol(0x1D6AE, "mbfEta", "mathalpha", "mathematical bold capital eta")
addSymbol(0x1D6AF, "mbfTheta", "mathalpha", "mathematical bold capital theta")
addSymbol(0x1D6B0, "mbfIota", "mathalpha", "mathematical bold capital iota")
addSymbol(0x1D6B1, "mbfKappa", "mathalpha", "mathematical bold capital kappa")
addSymbol(0x1D6B2, "mbfLambda", "mathalpha", "mathematical bold capital lambda")
addSymbol(0x1D6B3, "mbfMu", "mathalpha", "mathematical bold capital mu")
addSymbol(0x1D6B4, "mbfNu", "mathalpha", "mathematical bold capital nu")
addSymbol(0x1D6B5, "mbfXi", "mathalpha", "mathematical bold capital xi")
addSymbol(0x1D6B6, "mbfOmicron", "mathalpha", "mathematical bold capital omicron")
addSymbol(0x1D6B7, "mbfPi", "mathalpha", "mathematical bold capital pi")
addSymbol(0x1D6B8, "mbfRho", "mathalpha", "mathematical bold capital rho")
addSymbol(0x1D6B9, "mbfvarTheta", "mathalpha", "mathematical bold capital theta symbol")
addSymbol(0x1D6BA, "mbfSigma", "mathalpha", "mathematical bold capital sigma")
addSymbol(0x1D6BB, "mbfTau", "mathalpha", "mathematical bold capital tau")
addSymbol(0x1D6BC, "mbfUpsilon", "mathalpha", "mathematical bold capital upsilon")
addSymbol(0x1D6BD, "mbfPhi", "mathalpha", "mathematical bold capital phi")
addSymbol(0x1D6BE, "mbfChi", "mathalpha", "mathematical bold capital chi")
addSymbol(0x1D6BF, "mbfPsi", "mathalpha", "mathematical bold capital psi")
addSymbol(0x1D6C0, "mbfOmega", "mathalpha", "mathematical bold capital omega")
addSymbol(0x1D6C1, "mbfnabla", "mathalpha", "mathematical bold nabla")
addSymbol(0x1D6C2, "mbfalpha", "mathalpha", "mathematical bold small alpha")
addSymbol(0x1D6C3, "mbfbeta", "mathalpha", "mathematical bold small beta")
addSymbol(0x1D6C4, "mbfgamma", "mathalpha", "mathematical bold small gamma")
addSymbol(0x1D6C5, "mbfdelta", "mathalpha", "mathematical bold small delta")
addSymbol(0x1D6C6, "mbfvarepsilon", "mathalpha", "mathematical bold small varepsilon")
addSymbol(0x1D6C7, "mbfzeta", "mathalpha", "mathematical bold small zeta")
addSymbol(0x1D6C8, "mbfeta", "mathalpha", "mathematical bold small eta")
addSymbol(0x1D6C9, "mbftheta", "mathalpha", "mathematical bold small theta")
addSymbol(0x1D6CA, "mbfiota", "mathalpha", "mathematical bold small iota")
addSymbol(0x1D6CB, "mbfkappa", "mathalpha", "mathematical bold small kappa")
addSymbol(0x1D6CC, "mbflambda", "mathalpha", "mathematical bold small lambda")
addSymbol(0x1D6CD, "mbfmu", "mathalpha", "mathematical bold small mu")
addSymbol(0x1D6CE, "mbfnu", "mathalpha", "mathematical bold small nu")
addSymbol(0x1D6CF, "mbfxi", "mathalpha", "mathematical bold small xi")
addSymbol(0x1D6D0, "mbfomicron", "mathalpha", "mathematical bold small omicron")
addSymbol(0x1D6D1, "mbfpi", "mathalpha", "mathematical bold small pi")
addSymbol(0x1D6D2, "mbfrho", "mathalpha", "mathematical bold small rho")
addSymbol(0x1D6D3, "mbfvarsigma", "mathalpha", "mathematical bold small final sigma")
addSymbol(0x1D6D4, "mbfsigma", "mathalpha", "mathematical bold small sigma")
addSymbol(0x1D6D5, "mbftau", "mathalpha", "mathematical bold small tau")
addSymbol(0x1D6D6, "mbfupsilon", "mathalpha", "mathematical bold small upsilon")
addSymbol(0x1D6D7, "mbfvarphi", "mathalpha", "mathematical bold small phi")
addSymbol(0x1D6D8, "mbfchi", "mathalpha", "mathematical bold small chi")
addSymbol(0x1D6D9, "mbfpsi", "mathalpha", "mathematical bold small psi")
addSymbol(0x1D6DA, "mbfomega", "mathalpha", "mathematical bold small omega")
addSymbol(0x1D6DB, "mbfpartial", "mathalpha", "mathematical bold partial differential")
addSymbol(0x1D6DC, "mbfepsilon", "mathalpha", "mathematical bold varepsilon symbol")
addSymbol(0x1D6DD, "mbfvartheta", "mathalpha", "mathematical bold theta symbol")
addSymbol(0x1D6DE, "mbfvarkappa", "mathalpha", "mathematical bold kappa symbol")
addSymbol(0x1D6DF, "mbfphi", "mathalpha", "mathematical bold phi symbol")
addSymbol(0x1D6E0, "mbfvarrho", "mathalpha", "mathematical bold rho symbol")
addSymbol(0x1D6E1, "mbfvarpi", "mathalpha", "mathematical bold pi symbol")
addSymbol(0x1D6E2, "mitAlpha", "mathalpha", "mathematical italic capital alpha")
addSymbol(0x1D6E3, "mitBeta", "mathalpha", "mathematical italic capital beta")
addSymbol(0x1D6E4, "mitGamma", "mathalpha", "mathematical italic capital gamma")
addSymbol(0x1D6E5, "mitDelta", "mathalpha", "mathematical italic capital delta")
addSymbol(0x1D6E6, "mitEpsilon", "mathalpha", "mathematical italic capital epsilon")
addSymbol(0x1D6E7, "mitZeta", "mathalpha", "mathematical italic capital zeta")
addSymbol(0x1D6E8, "mitEta", "mathalpha", "mathematical italic capital eta")
addSymbol(0x1D6E9, "mitTheta", "mathalpha", "mathematical italic capital theta")
addSymbol(0x1D6EA, "mitIota", "mathalpha", "mathematical italic capital iota")
addSymbol(0x1D6EB, "mitKappa", "mathalpha", "mathematical italic capital kappa")
addSymbol(0x1D6EC, "mitLambda", "mathalpha", "mathematical italic capital lambda")
addSymbol(0x1D6ED, "mitMu", "mathalpha", "mathematical italic capital mu")
addSymbol(0x1D6EE, "mitNu", "mathalpha", "mathematical italic capital nu")
addSymbol(0x1D6EF, "mitXi", "mathalpha", "mathematical italic capital xi")
addSymbol(0x1D6F0, "mitOmicron", "mathalpha", "mathematical italic capital omicron")
addSymbol(0x1D6F1, "mitPi", "mathalpha", "mathematical italic capital pi")
addSymbol(0x1D6F2, "mitRho", "mathalpha", "mathematical italic capital rho")
addSymbol(0x1D6F3, "mitvarTheta", "mathalpha", "mathematical italic capital theta symbol")
addSymbol(0x1D6F4, "mitSigma", "mathalpha", "mathematical italic capital sigma")
addSymbol(0x1D6F5, "mitTau", "mathalpha", "mathematical italic capital tau")
addSymbol(0x1D6F6, "mitUpsilon", "mathalpha", "mathematical italic capital upsilon")
addSymbol(0x1D6F7, "mitPhi", "mathalpha", "mathematical italic capital phi")
addSymbol(0x1D6F8, "mitChi", "mathalpha", "mathematical italic capital chi")
addSymbol(0x1D6F9, "mitPsi", "mathalpha", "mathematical italic capital psi")
addSymbol(0x1D6FA, "mitOmega", "mathalpha", "mathematical italic capital omega")
addSymbol(0x1D6FB, "mitnabla", "mathalpha", "mathematical italic nabla")
addSymbol(0x1D6FC, "mitalpha", "mathalpha", "mathematical italic small alpha")
addSymbol(0x1D6FD, "mitbeta", "mathalpha", "mathematical italic small beta")
addSymbol(0x1D6FE, "mitgamma", "mathalpha", "mathematical italic small gamma")
addSymbol(0x1D6FF, "mitdelta", "mathalpha", "mathematical italic small delta")
addSymbol(0x1D700, "mitvarepsilon", "mathalpha", "mathematical italic small varepsilon")
addSymbol(0x1D701, "mitzeta", "mathalpha", "mathematical italic small zeta")
addSymbol(0x1D702, "miteta", "mathalpha", "mathematical italic small eta")
addSymbol(0x1D703, "mittheta", "mathalpha", "mathematical italic small theta")
addSymbol(0x1D704, "mitiota", "mathalpha", "mathematical italic small iota")
addSymbol(0x1D705, "mitkappa", "mathalpha", "mathematical italic small kappa")
addSymbol(0x1D706, "mitlambda", "mathalpha", "mathematical italic small lambda")
addSymbol(0x1D707, "mitmu", "mathalpha", "mathematical italic small mu")
addSymbol(0x1D708, "mitnu", "mathalpha", "mathematical italic small nu")
addSymbol(0x1D709, "mitxi", "mathalpha", "mathematical italic small xi")
addSymbol(0x1D70A, "mitomicron", "mathalpha", "mathematical italic small omicron")
addSymbol(0x1D70B, "mitpi", "mathalpha", "mathematical italic small pi")
addSymbol(0x1D70C, "mitrho", "mathalpha", "mathematical italic small rho")
addSymbol(0x1D70D, "mitvarsigma", "mathalpha", "mathematical italic small final sigma")
addSymbol(0x1D70E, "mitsigma", "mathalpha", "mathematical italic small sigma")
addSymbol(0x1D70F, "mittau", "mathalpha", "mathematical italic small tau")
addSymbol(0x1D710, "mitupsilon", "mathalpha", "mathematical italic small upsilon")
addSymbol(0x1D711, "mitvarphi", "mathalpha", "mathematical italic small phi")
addSymbol(0x1D712, "mitchi", "mathalpha", "mathematical italic small chi")
addSymbol(0x1D713, "mitpsi", "mathalpha", "mathematical italic small psi")
addSymbol(0x1D714, "mitomega", "mathalpha", "mathematical italic small omega")
addSymbol(0x1D715, "mitpartial", "mathalpha", "mathematical italic partial differential")
addSymbol(0x1D716, "mitepsilon", "mathalpha", "mathematical italic varepsilon symbol")
addSymbol(0x1D717, "mitvartheta", "mathalpha", "mathematical italic theta symbol")
addSymbol(0x1D718, "mitvarkappa", "mathalpha", "mathematical italic kappa symbol")
addSymbol(0x1D719, "mitphi", "mathalpha", "mathematical italic phi symbol")
addSymbol(0x1D71A, "mitvarrho", "mathalpha", "mathematical italic rho symbol")
addSymbol(0x1D71B, "mitvarpi", "mathalpha", "mathematical italic pi symbol")
addSymbol(0x1D71C, "mbfitAlpha", "mathalpha", "mathematical bold italic capital alpha")
addSymbol(0x1D71D, "mbfitBeta", "mathalpha", "mathematical bold italic capital beta")
addSymbol(0x1D71E, "mbfitGamma", "mathalpha", "mathematical bold italic capital gamma")
addSymbol(0x1D71F, "mbfitDelta", "mathalpha", "mathematical bold italic capital delta")
addSymbol(0x1D720, "mbfitEpsilon", "mathalpha", "mathematical bold italic capital epsilon")
addSymbol(0x1D721, "mbfitZeta", "mathalpha", "mathematical bold italic capital zeta")
addSymbol(0x1D722, "mbfitEta", "mathalpha", "mathematical bold italic capital eta")
addSymbol(0x1D723, "mbfitTheta", "mathalpha", "mathematical bold italic capital theta")
addSymbol(0x1D724, "mbfitIota", "mathalpha", "mathematical bold italic capital iota")
addSymbol(0x1D725, "mbfitKappa", "mathalpha", "mathematical bold italic capital kappa")
addSymbol(0x1D726, "mbfitLambda", "mathalpha", "mathematical bold italic capital lambda")
addSymbol(0x1D727, "mbfitMu", "mathalpha", "mathematical bold italic capital mu")
addSymbol(0x1D728, "mbfitNu", "mathalpha", "mathematical bold italic capital nu")
addSymbol(0x1D729, "mbfitXi", "mathalpha", "mathematical bold italic capital xi")
addSymbol(0x1D72A, "mbfitOmicron", "mathalpha", "mathematical bold italic capital omicron")
addSymbol(0x1D72B, "mbfitPi", "mathalpha", "mathematical bold italic capital pi")
addSymbol(0x1D72C, "mbfitRho", "mathalpha", "mathematical bold italic capital rho")
addSymbol(0x1D72D, "mbfitvarTheta", "mathalpha", "mathematical bold italic capital theta symbol")
addSymbol(0x1D72E, "mbfitSigma", "mathalpha", "mathematical bold italic capital sigma")
addSymbol(0x1D72F, "mbfitTau", "mathalpha", "mathematical bold italic capital tau")
addSymbol(0x1D730, "mbfitUpsilon", "mathalpha", "mathematical bold italic capital upsilon")
addSymbol(0x1D731, "mbfitPhi", "mathalpha", "mathematical bold italic capital phi")
addSymbol(0x1D732, "mbfitChi", "mathalpha", "mathematical bold italic capital chi")
addSymbol(0x1D733, "mbfitPsi", "mathalpha", "mathematical bold italic capital psi")
addSymbol(0x1D734, "mbfitOmega", "mathalpha", "mathematical bold italic capital omega")
addSymbol(0x1D735, "mbfitnabla", "mathalpha", "mathematical bold italic nabla")
addSymbol(0x1D736, "mbfitalpha", "mathalpha", "mathematical bold italic small alpha")
addSymbol(0x1D737, "mbfitbeta", "mathalpha", "mathematical bold italic small beta")
addSymbol(0x1D738, "mbfitgamma", "mathalpha", "mathematical bold italic small gamma")
addSymbol(0x1D739, "mbfitdelta", "mathalpha", "mathematical bold italic small delta")
addSymbol(0x1D73A, "mbfitvarepsilon", "mathalpha", "mathematical bold italic small varepsilon")
addSymbol(0x1D73B, "mbfitzeta", "mathalpha", "mathematical bold italic small zeta")
addSymbol(0x1D73C, "mbfiteta", "mathalpha", "mathematical bold italic small eta")
addSymbol(0x1D73D, "mbfittheta", "mathalpha", "mathematical bold italic small theta")
addSymbol(0x1D73E, "mbfitiota", "mathalpha", "mathematical bold italic small iota")
addSymbol(0x1D73F, "mbfitkappa", "mathalpha", "mathematical bold italic small kappa")
addSymbol(0x1D740, "mbfitlambda", "mathalpha", "mathematical bold italic small lambda")
addSymbol(0x1D741, "mbfitmu", "mathalpha", "mathematical bold italic small mu")
addSymbol(0x1D742, "mbfitnu", "mathalpha", "mathematical bold italic small nu")
addSymbol(0x1D743, "mbfitxi", "mathalpha", "mathematical bold italic small xi")
addSymbol(0x1D744, "mbfitomicron", "mathalpha", "mathematical bold italic small omicron")
addSymbol(0x1D745, "mbfitpi", "mathalpha", "mathematical bold italic small pi")
addSymbol(0x1D746, "mbfitrho", "mathalpha", "mathematical bold italic small rho")
addSymbol(0x1D747, "mbfitvarsigma", "mathalpha", "mathematical bold italic small final sigma")
addSymbol(0x1D748, "mbfitsigma", "mathalpha", "mathematical bold italic small sigma")
addSymbol(0x1D749, "mbfittau", "mathalpha", "mathematical bold italic small tau")
addSymbol(0x1D74A, "mbfitupsilon", "mathalpha", "mathematical bold italic small upsilon")
addSymbol(0x1D74B, "mbfitvarphi", "mathalpha", "mathematical bold italic small phi")
addSymbol(0x1D74C, "mbfitchi", "mathalpha", "mathematical bold italic small chi")
addSymbol(0x1D74D, "mbfitpsi", "mathalpha", "mathematical bold italic small psi")
addSymbol(0x1D74E, "mbfitomega", "mathalpha", "mathematical bold italic small omega")
addSymbol(0x1D74F, "mbfitpartial", "mathalpha", "mathematical bold italic partial differential")
addSymbol(0x1D750, "mbfitepsilon", "mathalpha", "mathematical bold italic varepsilon symbol")
addSymbol(0x1D751, "mbfitvartheta", "mathalpha", "mathematical bold italic theta symbol")
addSymbol(0x1D752, "mbfitvarkappa", "mathalpha", "mathematical bold italic kappa symbol")
addSymbol(0x1D753, "mbfitphi", "mathalpha", "mathematical bold italic phi symbol")
addSymbol(0x1D754, "mbfitvarrho", "mathalpha", "mathematical bold italic rho symbol")
addSymbol(0x1D755, "mbfitvarpi", "mathalpha", "mathematical bold italic pi symbol")
addSymbol(0x1D756, "mbfsansAlpha", "mathalpha", "mathematical sans-serif bold capital alpha")
addSymbol(0x1D757, "mbfsansBeta", "mathalpha", "mathematical sans-serif bold capital beta")
addSymbol(0x1D758, "mbfsansGamma", "mathalpha", "mathematical sans-serif bold capital gamma")
addSymbol(0x1D759, "mbfsansDelta", "mathalpha", "mathematical sans-serif bold capital delta")
addSymbol(0x1D75A, "mbfsansEpsilon", "mathalpha", "mathematical sans-serif bold capital epsilon")
addSymbol(0x1D75B, "mbfsansZeta", "mathalpha", "mathematical sans-serif bold capital zeta")
addSymbol(0x1D75C, "mbfsansEta", "mathalpha", "mathematical sans-serif bold capital eta")
addSymbol(0x1D75D, "mbfsansTheta", "mathalpha", "mathematical sans-serif bold capital theta")
addSymbol(0x1D75E, "mbfsansIota", "mathalpha", "mathematical sans-serif bold capital iota")
addSymbol(0x1D75F, "mbfsansKappa", "mathalpha", "mathematical sans-serif bold capital kappa")
addSymbol(0x1D760, "mbfsansLambda", "mathalpha", "mathematical sans-serif bold capital lambda")
addSymbol(0x1D761, "mbfsansMu", "mathalpha", "mathematical sans-serif bold capital mu")
addSymbol(0x1D762, "mbfsansNu", "mathalpha", "mathematical sans-serif bold capital nu")
addSymbol(0x1D763, "mbfsansXi", "mathalpha", "mathematical sans-serif bold capital xi")
addSymbol(0x1D764, "mbfsansOmicron", "mathalpha", "mathematical sans-serif bold capital omicron")
addSymbol(0x1D765, "mbfsansPi", "mathalpha", "mathematical sans-serif bold capital pi")
addSymbol(0x1D766, "mbfsansRho", "mathalpha", "mathematical sans-serif bold capital rho")
addSymbol(0x1D767, "mbfsansvarTheta", "mathalpha", "mathematical sans-serif bold capital theta symbol")
addSymbol(0x1D768, "mbfsansSigma", "mathalpha", "mathematical sans-serif bold capital sigma")
addSymbol(0x1D769, "mbfsansTau", "mathalpha", "mathematical sans-serif bold capital tau")
addSymbol(0x1D76A, "mbfsansUpsilon", "mathalpha", "mathematical sans-serif bold capital upsilon")
addSymbol(0x1D76B, "mbfsansPhi", "mathalpha", "mathematical sans-serif bold capital phi")
addSymbol(0x1D76C, "mbfsansChi", "mathalpha", "mathematical sans-serif bold capital chi")
addSymbol(0x1D76D, "mbfsansPsi", "mathalpha", "mathematical sans-serif bold capital psi")
addSymbol(0x1D76E, "mbfsansOmega", "mathalpha", "mathematical sans-serif bold capital omega")
addSymbol(0x1D76F, "mbfsansnabla", "mathalpha", "mathematical sans-serif bold nabla")
addSymbol(0x1D770, "mbfsansalpha", "mathalpha", "mathematical sans-serif bold small alpha")
addSymbol(0x1D771, "mbfsansbeta", "mathalpha", "mathematical sans-serif bold small beta")
addSymbol(0x1D772, "mbfsansgamma", "mathalpha", "mathematical sans-serif bold small gamma")
addSymbol(0x1D773, "mbfsansdelta", "mathalpha", "mathematical sans-serif bold small delta")
addSymbol(0x1D774, "mbfsansvarepsilon", "mathalpha", "mathematical sans-serif bold small varepsilon")
addSymbol(0x1D775, "mbfsanszeta", "mathalpha", "mathematical sans-serif bold small zeta")
addSymbol(0x1D776, "mbfsanseta", "mathalpha", "mathematical sans-serif bold small eta")
addSymbol(0x1D777, "mbfsanstheta", "mathalpha", "mathematical sans-serif bold small theta")
addSymbol(0x1D778, "mbfsansiota", "mathalpha", "mathematical sans-serif bold small iota")
addSymbol(0x1D779, "mbfsanskappa", "mathalpha", "mathematical sans-serif bold small kappa")
addSymbol(0x1D77A, "mbfsanslambda", "mathalpha", "mathematical sans-serif bold small lambda")
addSymbol(0x1D77B, "mbfsansmu", "mathalpha", "mathematical sans-serif bold small mu")
addSymbol(0x1D77C, "mbfsansnu", "mathalpha", "mathematical sans-serif bold small nu")
addSymbol(0x1D77D, "mbfsansxi", "mathalpha", "mathematical sans-serif bold small xi")
addSymbol(0x1D77E, "mbfsansomicron", "mathalpha", "mathematical sans-serif bold small omicron")
addSymbol(0x1D77F, "mbfsanspi", "mathalpha", "mathematical sans-serif bold small pi")
addSymbol(0x1D780, "mbfsansrho", "mathalpha", "mathematical sans-serif bold small rho")
addSymbol(0x1D781, "mbfsansvarsigma", "mathalpha", "mathematical sans-serif bold small final sigma")
addSymbol(0x1D782, "mbfsanssigma", "mathalpha", "mathematical sans-serif bold small sigma")
addSymbol(0x1D783, "mbfsanstau", "mathalpha", "mathematical sans-serif bold small tau")
addSymbol(0x1D784, "mbfsansupsilon", "mathalpha", "mathematical sans-serif bold small upsilon")
addSymbol(0x1D785, "mbfsansvarphi", "mathalpha", "mathematical sans-serif bold small phi")
addSymbol(0x1D786, "mbfsanschi", "mathalpha", "mathematical sans-serif bold small chi")
addSymbol(0x1D787, "mbfsanspsi", "mathalpha", "mathematical sans-serif bold small psi")
addSymbol(0x1D788, "mbfsansomega", "mathalpha", "mathematical sans-serif bold small omega")
addSymbol(0x1D789, "mbfsanspartial", "mathalpha", "mathematical sans-serif bold partial differential")
addSymbol(0x1D78A, "mbfsansepsilon", "mathalpha", "mathematical sans-serif bold varepsilon symbol")
addSymbol(0x1D78B, "mbfsansvartheta", "mathalpha", "mathematical sans-serif bold theta symbol")
addSymbol(0x1D78C, "mbfsansvarkappa", "mathalpha", "mathematical sans-serif bold kappa symbol")
addSymbol(0x1D78D, "mbfsansphi", "mathalpha", "mathematical sans-serif bold phi symbol")
addSymbol(0x1D78E, "mbfsansvarrho", "mathalpha", "mathematical sans-serif bold rho symbol")
addSymbol(0x1D78F, "mbfsansvarpi", "mathalpha", "mathematical sans-serif bold pi symbol")
addSymbol(0x1D790, "mbfitsansAlpha", "mathalpha", "mathematical sans-serif bold italic capital alpha")
addSymbol(0x1D791, "mbfitsansBeta", "mathalpha", "mathematical sans-serif bold italic capital beta")
addSymbol(0x1D792, "mbfitsansGamma", "mathalpha", "mathematical sans-serif bold italic capital gamma")
addSymbol(0x1D793, "mbfitsansDelta", "mathalpha", "mathematical sans-serif bold italic capital delta")
addSymbol(0x1D794, "mbfitsansEpsilon", "mathalpha", "mathematical sans-serif bold italic capital epsilon")
addSymbol(0x1D795, "mbfitsansZeta", "mathalpha", "mathematical sans-serif bold italic capital zeta")
addSymbol(0x1D796, "mbfitsansEta", "mathalpha", "mathematical sans-serif bold italic capital eta")
addSymbol(0x1D797, "mbfitsansTheta", "mathalpha", "mathematical sans-serif bold italic capital theta")
addSymbol(0x1D798, "mbfitsansIota", "mathalpha", "mathematical sans-serif bold italic capital iota")
addSymbol(0x1D799, "mbfitsansKappa", "mathalpha", "mathematical sans-serif bold italic capital kappa")
addSymbol(0x1D79A, "mbfitsansLambda", "mathalpha", "mathematical sans-serif bold italic capital lambda")
addSymbol(0x1D79B, "mbfitsansMu", "mathalpha", "mathematical sans-serif bold italic capital mu")
addSymbol(0x1D79C, "mbfitsansNu", "mathalpha", "mathematical sans-serif bold italic capital nu")
addSymbol(0x1D79D, "mbfitsansXi", "mathalpha", "mathematical sans-serif bold italic capital xi")
addSymbol(0x1D79E, "mbfitsansOmicron", "mathalpha", "mathematical sans-serif bold italic capital omicron")
addSymbol(0x1D79F, "mbfitsansPi", "mathalpha", "mathematical sans-serif bold italic capital pi")
addSymbol(0x1D7A0, "mbfitsansRho", "mathalpha", "mathematical sans-serif bold italic capital rho")
addSymbol(0x1D7A1, "mbfitsansvarTheta", "mathalpha", "mathematical sans-serif bold italic capital theta symbol")
addSymbol(0x1D7A2, "mbfitsansSigma", "mathalpha", "mathematical sans-serif bold italic capital sigma")
addSymbol(0x1D7A3, "mbfitsansTau", "mathalpha", "mathematical sans-serif bold italic capital tau")
addSymbol(0x1D7A4, "mbfitsansUpsilon", "mathalpha", "mathematical sans-serif bold italic capital upsilon")
addSymbol(0x1D7A5, "mbfitsansPhi", "mathalpha", "mathematical sans-serif bold italic capital phi")
addSymbol(0x1D7A6, "mbfitsansChi", "mathalpha", "mathematical sans-serif bold italic capital chi")
addSymbol(0x1D7A7, "mbfitsansPsi", "mathalpha", "mathematical sans-serif bold italic capital psi")
addSymbol(0x1D7A8, "mbfitsansOmega", "mathalpha", "mathematical sans-serif bold italic capital omega")
addSymbol(0x1D7A9, "mbfitsansnabla", "mathalpha", "mathematical sans-serif bold italic nabla")
addSymbol(0x1D7AA, "mbfitsansalpha", "mathalpha", "mathematical sans-serif bold italic small alpha")
addSymbol(0x1D7AB, "mbfitsansbeta", "mathalpha", "mathematical sans-serif bold italic small beta")
addSymbol(0x1D7AC, "mbfitsansgamma", "mathalpha", "mathematical sans-serif bold italic small gamma")
addSymbol(0x1D7AD, "mbfitsansdelta", "mathalpha", "mathematical sans-serif bold italic small delta")
addSymbol(0x1D7AE, "mbfitsansvarepsilon", "mathalpha", "mathematical sans-serif bold italic small varepsilon")
addSymbol(0x1D7AF, "mbfitsanszeta", "mathalpha", "mathematical sans-serif bold italic small zeta")
addSymbol(0x1D7B0, "mbfitsanseta", "mathalpha", "mathematical sans-serif bold italic small eta")
addSymbol(0x1D7B1, "mbfitsanstheta", "mathalpha", "mathematical sans-serif bold italic small theta")
addSymbol(0x1D7B2, "mbfitsansiota", "mathalpha", "mathematical sans-serif bold italic small iota")
addSymbol(0x1D7B3, "mbfitsanskappa", "mathalpha", "mathematical sans-serif bold italic small kappa")
addSymbol(0x1D7B4, "mbfitsanslambda", "mathalpha", "mathematical sans-serif bold italic small lambda")
addSymbol(0x1D7B5, "mbfitsansmu", "mathalpha", "mathematical sans-serif bold italic small mu")
addSymbol(0x1D7B6, "mbfitsansnu", "mathalpha", "mathematical sans-serif bold italic small nu")
addSymbol(0x1D7B7, "mbfitsansxi", "mathalpha", "mathematical sans-serif bold italic small xi")
addSymbol(0x1D7B8, "mbfitsansomicron", "mathalpha", "mathematical sans-serif bold italic small omicron")
addSymbol(0x1D7B9, "mbfitsanspi", "mathalpha", "mathematical sans-serif bold italic small pi")
addSymbol(0x1D7BA, "mbfitsansrho", "mathalpha", "mathematical sans-serif bold italic small rho")
addSymbol(0x1D7BB, "mbfitsansvarsigma", "mathalpha", "mathematical sans-serif bold italic small final sigma")
addSymbol(0x1D7BC, "mbfitsanssigma", "mathalpha", "mathematical sans-serif bold italic small sigma")
addSymbol(0x1D7BD, "mbfitsanstau", "mathalpha", "mathematical sans-serif bold italic small tau")
addSymbol(0x1D7BE, "mbfitsansupsilon", "mathalpha", "mathematical sans-serif bold italic small upsilon")
addSymbol(0x1D7BF, "mbfitsansvarphi", "mathalpha", "mathematical sans-serif bold italic small phi")
addSymbol(0x1D7C0, "mbfitsanschi", "mathalpha", "mathematical sans-serif bold italic small chi")
addSymbol(0x1D7C1, "mbfitsanspsi", "mathalpha", "mathematical sans-serif bold italic small psi")
addSymbol(0x1D7C2, "mbfitsansomega", "mathalpha", "mathematical sans-serif bold italic small omega")
addSymbol(0x1D7C3, "mbfitsanspartial", "mathalpha", "mathematical sans-serif bold italic partial differential")
addSymbol(0x1D7C4, "mbfitsansepsilon", "mathalpha", "mathematical sans-serif bold italic varepsilon symbol")
addSymbol(0x1D7C5, "mbfitsansvartheta", "mathalpha", "mathematical sans-serif bold italic theta symbol")
addSymbol(0x1D7C6, "mbfitsansvarkappa", "mathalpha", "mathematical sans-serif bold italic kappa symbol")
addSymbol(0x1D7C7, "mbfitsansphi", "mathalpha", "mathematical sans-serif bold italic phi symbol")
addSymbol(0x1D7C8, "mbfitsansvarrho", "mathalpha", "mathematical sans-serif bold italic rho symbol")
addSymbol(0x1D7C9, "mbfitsansvarpi", "mathalpha", "mathematical sans-serif bold italic pi symbol")
addSymbol(0x1D7CA, "mbfDigamma", "mathalpha", "mathematical bold capital digamma")
addSymbol(0x1D7CB, "mbfdigamma", "mathalpha", "mathematical bold small digamma")
addSymbol(0x1D7CE, "mbfzero", "mathord", "mathematical bold digit 0")
addSymbol(0x1D7CF, "mbfone", "mathord", "mathematical bold digit 1")
addSymbol(0x1D7D0, "mbftwo", "mathord", "mathematical bold digit 2")
addSymbol(0x1D7D1, "mbfthree", "mathord", "mathematical bold digit 3")
addSymbol(0x1D7D2, "mbffour", "mathord", "mathematical bold digit 4")
addSymbol(0x1D7D3, "mbffive", "mathord", "mathematical bold digit 5")
addSymbol(0x1D7D4, "mbfsix", "mathord", "mathematical bold digit 6")
addSymbol(0x1D7D5, "mbfseven", "mathord", "mathematical bold digit 7")
addSymbol(0x1D7D6, "mbfeight", "mathord", "mathematical bold digit 8")
addSymbol(0x1D7D7, "mbfnine", "mathord", "mathematical bold digit 9")
addSymbol(0x1D7D8, "Bbbzero", "mathord", "mathematical double-struck digit 0")
addSymbol(0x1D7D9, "Bbbone", "mathord", "mathematical double-struck digit 1")
addSymbol(0x1D7DA, "Bbbtwo", "mathord", "mathematical double-struck digit 2")
addSymbol(0x1D7DB, "Bbbthree", "mathord", "mathematical double-struck digit 3")
addSymbol(0x1D7DC, "Bbbfour", "mathord", "mathematical double-struck digit 4")
addSymbol(0x1D7DD, "Bbbfive", "mathord", "mathematical double-struck digit 5")
addSymbol(0x1D7DE, "Bbbsix", "mathord", "mathematical double-struck digit 6")
addSymbol(0x1D7DF, "Bbbseven", "mathord", "mathematical double-struck digit 7")
addSymbol(0x1D7E0, "Bbbeight", "mathord", "mathematical double-struck digit 8")
addSymbol(0x1D7E1, "Bbbnine", "mathord", "mathematical double-struck digit 9")
addSymbol(0x1D7E2, "msanszero", "mathord", "mathematical sans-serif digit 0")
addSymbol(0x1D7E3, "msansone", "mathord", "mathematical sans-serif digit 1")
addSymbol(0x1D7E4, "msanstwo", "mathord", "mathematical sans-serif digit 2")
addSymbol(0x1D7E5, "msansthree", "mathord", "mathematical sans-serif digit 3")
addSymbol(0x1D7E6, "msansfour", "mathord", "mathematical sans-serif digit 4")
addSymbol(0x1D7E7, "msansfive", "mathord", "mathematical sans-serif digit 5")
addSymbol(0x1D7E8, "msanssix", "mathord", "mathematical sans-serif digit 6")
addSymbol(0x1D7E9, "msansseven", "mathord", "mathematical sans-serif digit 7")
addSymbol(0x1D7EA, "msanseight", "mathord", "mathematical sans-serif digit 8")
addSymbol(0x1D7EB, "msansnine", "mathord", "mathematical sans-serif digit 9")
addSymbol(0x1D7EC, "mbfsanszero", "mathord", "mathematical sans-serif bold digit 0")
addSymbol(0x1D7ED, "mbfsansone", "mathord", "mathematical sans-serif bold digit 1")
addSymbol(0x1D7EE, "mbfsanstwo", "mathord", "mathematical sans-serif bold digit 2")
addSymbol(0x1D7EF, "mbfsansthree", "mathord", "mathematical sans-serif bold digit 3")
addSymbol(0x1D7F0, "mbfsansfour", "mathord", "mathematical sans-serif bold digit 4")
addSymbol(0x1D7F1, "mbfsansfive", "mathord", "mathematical sans-serif bold digit 5")
addSymbol(0x1D7F2, "mbfsanssix", "mathord", "mathematical sans-serif bold digit 6")
addSymbol(0x1D7F3, "mbfsansseven", "mathord", "mathematical sans-serif bold digit 7")
addSymbol(0x1D7F4, "mbfsanseight", "mathord", "mathematical sans-serif bold digit 8")
addSymbol(0x1D7F5, "mbfsansnine", "mathord", "mathematical sans-serif bold digit 9")
addSymbol(0x1D7F6, "mttzero", "mathord", "mathematical monospace digit 0")
addSymbol(0x1D7F7, "mttone", "mathord", "mathematical monospace digit 1")
addSymbol(0x1D7F8, "mtttwo", "mathord", "mathematical monospace digit 2")
addSymbol(0x1D7F9, "mttthree", "mathord", "mathematical monospace digit 3")
addSymbol(0x1D7FA, "mttfour", "mathord", "mathematical monospace digit 4")
addSymbol(0x1D7FB, "mttfive", "mathord", "mathematical monospace digit 5")
addSymbol(0x1D7FC, "mttsix", "mathord", "mathematical monospace digit 6")
addSymbol(0x1D7FD, "mttseven", "mathord", "mathematical monospace digit 7")
addSymbol(0x1D7FE, "mtteight", "mathord", "mathematical monospace digit 8")
addSymbol(0x1D7FF, "mttnine", "mathord", "mathematical monospace digit 9")
addSymbol(0x1EEF0, "arabicmaj", "mathop", "arabic mathematical operator meem with hah with tatweel")
addSymbol(0x1EEF1, "arabichad", "mathop", "arabic mathematical operator hah with dal")

-- Operators
symbols["in"] = "∈"
symbols.sum = "∑"
symbols.infty = "∞"
symbols.dots = "⋯"
symbols.implies = "⟹"

-- Greek letters
symbols.alpha = "α"
symbols.beta = "β"
symbols.gamma = "γ"
symbols.delta = "δ"
symbols.epsilon = "ϵ"
symbols.varepsilon = "ε"
symbols.zeta = "ζ"
symbols.eta = "η"
symbols.theta = "θ"
symbols.vartheta = "ϑ"
symbols.iota = "ι"
symbols.kappa = "κ"
symbols.lambda = "λ"
symbols.mu = "μ"
symbols.nu = "ν"
symbols.xi = "ξ"
symbols.omicron = "ο"
symbols.pi = "π"
symbols.varpi = "ϖ"
symbols.rho = "ρ"
symbols.varrho = "ϱ"
symbols.sigma = "σ"
symbols.varsigma = "ς"
symbols.tau = "τ"
symbols.upsilon = "υ"
symbols.phi = "ϕ"
symbols.varphi = "φ"
symbols.chi = "χ"
symbols.psi = "ψ"
symbols.omega = "ω"
symbols.Alpha = "Α"
symbols.Beta = "Β"
symbols.Gamma = "Γ"
symbols.Delta = "Δ"
symbols.Epsilon = "Ε"
symbols.Zeta = "Ζ"
symbols.Eta = "Η"
symbols.Theta = "Θ"
symbols.Iota = "Ι"
symbols.Kappa = "Κ"
symbols.Lambda = "Λ"
symbols.Mu = "Μ"
symbols.Nu = "Ν"
symbols.Xi = "Ξ"
symbols.Omicron = "Ο"
symbols.Pi = "Π"
symbols.Rho = "Ρ"
symbols.Sigma = "Σ"
symbols.Tau = "Τ"
symbols.Upsilon = "Υ"
symbols.Phi = "Φ"
symbols.Chi = "Χ"
symbols.Psi = "Ψ"
symbols.Omega = "Ω"

symbolDefaults["+"] = { atom = atomType.binaryOperator }
symbolDefaults["−"] = { atom = atomType.binaryOperator }
symbolDefaults["<"] = { atom = atomType.relationalOperator }
symbolDefaults["⩽"] = { atom = atomType.relationalOperator }
symbolDefaults[">"] = { atom = atomType.relationalOperator }
symbolDefaults["⩾"] = { atom = atomType.relationalOperator }
symbolDefaults["="] = { atom = atomType.relationalOperator }
symbolDefaults["≠"] = { atom = atomType.relationalOperator }
symbolDefaults["∈"] = { atom = atomType.relationalOperator }
symbolDefaults["⊆"] = { atom = atomType.relationalOperator }
symbolDefaults["∑"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["∏"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["⋀"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["⋁"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["⋂"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["⋃"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["⨅"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["⨆"] = { atom = atomType.bigOperator, largeop = true, movablelimits = true }
symbolDefaults["∫"] = { largeop = true }
symbolDefaults["∬"] = { largeop = true }
symbolDefaults["∭"] = { largeop = true }
symbolDefaults["∮"] = { largeop = true }
symbolDefaults["∯"] = { largeop = true }
symbolDefaults["∰"] = { largeop = true }
symbolDefaults[","] = { atom = atomType.punctuationSymbol }
symbolDefaults["⟹"] = { atom = atomType.relationalOperator }
symbolDefaults["/"] = { atom = atomType.binaryOperator }
symbolDefaults[":"] = { atom = atomType.relationalOperator }
symbolDefaults["⟶"] = { atom = atomType.relationalOperator }
symbolDefaults["|"] = { atom = atomType.relationalOperator }
symbolDefaults["("] = { atom = atomType.openingSymbol, stretchy = true }
symbolDefaults[")"] = { atom = atomType.closeSymbol, stretchy = true }
symbolDefaults["["] = { atom = atomType.openingSymbol, stretchy = true }
symbolDefaults["]"] = { atom = atomType.closeSymbol, stretchy = true }
symbolDefaults["{"] = { atom = atomType.openingSymbol, stretchy = true }
symbolDefaults["}"] = { atom = atomType.closeSymbol, stretchy = true }
symbolDefaults["mod"] = { atom = atomType.binaryOperator }
symbolDefaults["%"] = { atom = atomType.binaryOperator }

return {
   symbols = symbols,
   atomType = atomType,
   symbolDefaults = symbolDefaults,
}
