# DOSBox - Print the path to the disk image
# An empty string is returned if no disk image is required for the current game.
# USAGE: dosbox_disk_image_path
dosbox_disk_image_path() {
	local disk_image_path
	disk_image_path=$(context_value 'DOSBOX_DISK_IMAGE_PATH')
	## Check for a disk image set using the legacy variable
	if [ -z "$disk_image_path" ] && [ -n "${GAME_IMAGE:-}" ]; then
		if compatibility_level_is_at_least '2.32'; then
			warning_deprecated_variable 'GAME_IMAGE' 'DOSBOX_DISK_IMAGE_PATH'
		fi
		disk_image_path="$GAME_IMAGE"
	fi

	printf '%s' "$disk_image_path"
}

# DOSBox - Print the type of the disk image
# USAGE: dosbox_disk_image_type
dosbox_disk_image_type() {
	local disk_image_type
	disk_image_type=$(context_value 'DOSBOX_DISK_IMAGE_TYPE')
	## Check for a disk image set using the legacy variable
	if [ -z "$disk_image_type" ] && [ -n "${GAME_IMAGE_TYPE:-}" ]; then
		if compatibility_level_is_at_least '2.32'; then
			warning_deprecated_variable 'GAME_IMAGE_TYPE' 'DOSBOX_DISK_IMAGE_TYPE'
		fi
		disk_image_type="$GAME_IMAGE_TYPE"
	fi
	## Fall back on the default type
	if [ -z "$disk_image_type" ]; then
		disk_image_type='iso'
	fi

	printf '%s' "$disk_image_type"
}

# DOSBox - Compute the full path of the disk image
# An empty string is returned if the disk image could not be found.
# USAGE: disk_image_path_full
disk_image_path_full() {
	# Look for the disk image in the current package
	local disk_image_path_full disk_image_path package package_path path_game_data
	disk_image_path=$(dosbox_disk_image_path)
	package=$(current_package)
	package_path=$(package_path "$package")
	path_game_data=$(path_game_data)
	disk_image_path_full="${package_path}${path_game_data}/${disk_image_path}"
	if [ -e "$disk_image_path_full" ]; then
		printf '%s' "$disk_image_path_full"
		return 0
	fi

	# Look for the disk image in all packages
	local packages_list
	packages_list=$(packages_list)
	for package in $packages_list; do
		package_path=$(package_path "$package")
		disk_image_path_full="${package_path}${path_game_data}/${disk_image_path}"
		if [ -e "$disk_image_path_full" ]; then
			printf '%s' "$disk_image_path_full"
			return 0
		fi
	done

	# Nothing is printed if the disk image could not be found.
}

# DOSBox - Check for the presence of the disk image
# USAGE: dosbox_image_presence_check
dosbox_image_presence_check() {
	local disk_image_path_full
	disk_image_path_full=$(disk_image_path_full)

	test -n "$disk_image_path_full"
}

# DOSBox - If a disk image is required, print the command mounting it
# USAGE: dosbox_image_mount
dosbox_image_mount() {
	local disk_image_path
	disk_image_path=$(dosbox_disk_image_path)

	# Return early if no disk image is required by the current game
	if [ -z "$disk_image_path" ]; then
		return 0
	fi

	# Check for the presence of the disk image
	if ! dosbox_image_presence_check; then
		error_dosbox_disk_image_no_found "$disk_image_path"
		return 1
	fi

	# Print the command used to mount the disk image, based on its type
	local disk_image_type
	disk_image_type=$(dosbox_disk_image_type)
	case "$disk_image_type" in
		('cdrom')
			local disk_image_path_full
			disk_image_path_full=$(disk_image_path_full)
			if [ -d "$disk_image_path_full" ]; then
				printf 'mount d %s -t cdrom' "$disk_image_path"
			else
				printf 'imgmount d %s -t cdrom' "$disk_image_path"
			fi
		;;
		('iso')
			printf 'imgmount d %s -t iso -fs iso' "$disk_image_path"
		;;
		(*)
			error_dosbox_disk_image_type_invalid "$disk_image_path" "$disk_image_type"
			return 1
		;;
	esac
	printf '\n'
}

