/*
 *
 *  OBEX Server
 *
 *  Copyright (C) 2007-2008  Nokia Corporation
 *  Copyright (C) 2007-2008  Instituto Nokia de Tecnologia (INdT)
 *  Copyright (C) 2007-2009  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/rfcomm.h>

#include <glib.h>

#include <openobex/obex.h>
#include <openobex/obex_const.h>

#include "logging.h"
#include "bluetooth.h"
#include "obex.h"
#include "dbus.h"

#define BT_RX_MTU 32767
#define BT_TX_MTU 32767

static GSList *servers = NULL;

static gboolean connect_event(GIOChannel *io, GIOCondition cond, gpointer user_data)
{
	struct sockaddr_rc raddr;
	socklen_t alen;
	struct server *server = user_data;
	gchar address[18];
	gint err, sk, nsk;

	sk = g_io_channel_unix_get_fd(io);
	alen = sizeof(raddr);
	nsk = accept(sk, (struct sockaddr *) &raddr, &alen);
	if (nsk < 0)
		return TRUE;

	alen = sizeof(raddr);
	if (getpeername(nsk, (struct sockaddr *) &raddr, &alen) < 0) {
		err = errno;
		error("getpeername(): %s(%d)", strerror(err), err);
		close(nsk);
		return TRUE;
	}

	ba2str(&raddr.rc_bdaddr, address);
	info("New connection from: %s, channel %u, fd %d", address,
			raddr.rc_channel, nsk);

	if (server->services != OBEX_OPP) {
		if (request_service_authorization(server, nsk) < 0)
			close(nsk);

		return TRUE;
	}

	if (obex_session_start(nsk, server) < 0)
		close(nsk);

	return TRUE;
}

static gint server_start(struct server *server)
{
	struct sockaddr_rc laddr;
	int err, sk, arg;

	sk = socket(AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
	if (sk < 0) {
		err = errno;
		error("socket(): %s(%d)", strerror(err), err);
		return -err;
	}

	arg = fcntl(sk, F_GETFL);
	if (arg < 0) {
		err = errno;
		goto failed;
	}

	arg |= O_NONBLOCK;
	if (fcntl(sk, F_SETFL, arg) < 0) {
		err = errno;
		goto failed;
	}

	if (server->secure) {
		int lm = RFCOMM_LM_AUTH | RFCOMM_LM_ENCRYPT;

		if (setsockopt(sk, SOL_RFCOMM, RFCOMM_LM, &lm, sizeof(lm)) < 0) {
			err = errno;
			goto failed;
		}
	}

	memset(&laddr, 0, sizeof(laddr));
	laddr.rc_family = AF_BLUETOOTH;
	bacpy(&laddr.rc_bdaddr, BDADDR_ANY);
	laddr.rc_channel = server->channel;

	if (bind(sk, (struct sockaddr *) &laddr, sizeof(laddr)) < 0) {
		err = errno;
		goto failed;
	}

	if (listen(sk, 10) < 0) {
		err = errno;
		goto failed;
	}

	server->io = g_io_channel_unix_new(sk);
	g_io_channel_set_close_on_unref(server->io, TRUE);
	server->watch = g_io_add_watch(server->io,
				G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
				connect_event, server);

	return 0;

failed:
	error("Bluetooth server register failed: %s(%d)", strerror(err), err);
	close(sk);

	return -err;
}

static gint server_stop(struct server *server)
{
	if (!server->io)
		return -EINVAL;

	if (server->watch) {
		g_source_remove(server->watch);
		server->watch = 0;
	}

	g_io_channel_unref(server->io);
	server->io = NULL;

	return 0;
}

static gint server_register(guint16 service, const gchar *name, guint8 channel,
			const gchar *folder, gboolean secure,
			gboolean auto_accept, const gchar *capability)
{
	struct server *server;
	int err;

	server = g_new0(struct server, 1);
	server->services = service;
	server->name = g_strdup(name);
	server->folder = g_strdup(folder);
	server->auto_accept = auto_accept;
	server->capability = g_strdup(capability);
	server->channel = channel;
	server->secure = secure;
	server->rx_mtu = BT_RX_MTU;
	server->tx_mtu = BT_TX_MTU;

	err = server_start(server);
	if (err < 0) {
		server_free(server);
		return err;
	}

	servers = g_slist_append(servers, server);

	register_record(server, NULL);

	return 0;
}

gint bluetooth_init(guint service, const gchar *name, const gchar *folder,
				guint8 channel, gboolean secure,
				gboolean auto_accept, const gchar *capability)
{
	return server_register(service, name, channel, folder,
					secure, auto_accept, capability);
}

void bluetooth_exit(void)
{
	return;
}

void bluetooth_start(void)
{
	GSList *l;

	for (l = servers; l; l = l->next) {
		struct server *server = l->data;

		if (server_start(server) < 0)
			continue;

		register_record(server, NULL);
	}
}

void bluetooth_stop(void)
{
	GSList *l;

	for (l = servers; l; l = l->next) {
		struct server *server = l->data;

		server_stop(server);
	}
}
