/*
 * $Id: libcap.h,v 1.2 1997/04/28 00:57:11 morgan Exp $
 *
 * Copyright (c) 1997 Andrew G Morgan <morgan@parc.power.net>
 *
 * See end of file for Log.
 *
 * This file contains internal definitions for the various functions in
 * this small capability library.
 */

#ifndef LIBCAP_H
#define LIBCAP_H

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/capability.h>

/*
 * This is a pointer to a struct containing three consecutive
 * capability sets in the order of the cap_flag_t type: the are
 * effective,inheritable and permitted.  This is the type that the
 * user-space routines think of as 'internal' capabilities - this is
 * the type that is passed to the kernel with the system calls related
 * to processes.
 */

#define CAP_T_MAGIC 0xCA90D0
struct _cap_struct {
    int magic;
    struct __cap_s set[3];
};

/*
 * Do we match the local kernel?
 */

#if !defined(_LINUX_CAPABILITY_VERSION) || \
            (_LINUX_CAPABILITY_VERSION != 0x19970420)

# error "Kernel <linux/capability.h> does not match library"
# error "file "libcap.h" --> fix and recompile libcap"

#endif

/*
 * Private definitions for internal use by the library.
 */

#define good_cap_t(c)      ((c) && (c)->magic == CAP_T_MAGIC)

/*
 * library debugging
 */
#ifdef DEBUG

#include <stdio.h>
# define _cap_debug(f, x...)  { \
    fprintf(stderr, __FUNCTION__ "(" __FILE__ ":%d): ", __LINE__); \
    fprintf(stderr, f, ## x); \
    fprintf(stderr, "\n"); \
}
# define _cap_debugcap(s, c) \
    fprintf(stderr, __FUNCTION__ "(" __FILE__ ":%d): " s \
       "%08x %08x %08x %08x\n", __LINE__, \
       (c)->_blk[0], (c)->_blk[1], (c)->_blk[2], (c)->_blk[3])

#else /* !DEBUG */

# define _cap_debug(f, x...)
# define _cap_debugcap(s, c)

#endif /* DEBUG */

/*
 * These are semi-public prototypes, they will only be defined in
 * <sys/capability.h> if _POSIX_SOURCE is not #define'd, so we
 * place them here too.
 */

int _setproccap(size_t, __cap_s const *,__cap_s const *, __cap_s const *);
int _getproccap(size_t, __cap_s *,__cap_s *, __cap_s *);
int _setfilecap(char const *, size_t, __cap_s const *,
		__cap_s const *, __cap_s const *);
int _getfilecap(char const *, size_t, __cap_s *, __cap_s *, __cap_s *);
int _fsetfilecap(int, size_t, __cap_s const *,
		__cap_s const *, __cap_s const *);
int _fgetfilecap(int, size_t, __cap_s *, __cap_s *, __cap_s *);

extern char const *_cap_names[__CAP_BITS];

#endif /* LIBCAP_H */

/*
 * $Log: libcap.h,v $
 * Revision 1.2  1997/04/28 00:57:11  morgan
 * zefram's replacement file with a number of bug fixes from AGM
 *
 * Revision 1.1  1997/04/21 04:32:52  morgan
 * Initial revision
 *
 */
