/* Copyright (C) 2004 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#ifdef USE_SLP

#define _GNU_SOURCE

#include <netdb.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <slp.h>

#include "use_slp.h"
#include "dbg_log.h"

static void
pwdutilsSLPRegReport (SLPHandle hslp __attribute__ ((unused)),
		      SLPError errcode, void* cookie)
{
  /* return the error code in the cookie */
  *(SLPError*)cookie = errcode;
}

/* the URL we use to register.  */

static char *url = NULL;

int
register_slp (int port)
{
  SLPError err;
  SLPError callbackerr;
  SLPHandle hslp;
  char hostname[1024];
  char *hname;
  struct hostent *hp = NULL;

  if (url != NULL)
    {
      dbg_log ("URL already registerd!");
      return -1;
    }

  gethostname (hostname, sizeof (hostname));
  if (isdigit (hostname[0]))
    {
      char addr[INADDRSZ];
      if (inet_pton (AF_INET, hostname, &addr))
        hp = gethostbyaddr (addr, sizeof (addr), AF_INET);
    }
  else
    hp = gethostbyname (hostname);
  hname = hp->h_name;

  if (asprintf (&url, "service:rpasswdd://%s:%i/", hname, port) < 0)
    {
      dbg_log ("Out of memory");
      return -1;
    }

  err = SLPOpen ("en", SLP_FALSE, &hslp);
  if(err != SLP_OK)
    {
      dbg_log ("Error opening slp handle %i", err);
      return -1;
    }

    /* Register a service with SLP */
  err = SLPReg (hslp, url, SLP_LIFETIME_MAXIMUM, 0,
		"",
		SLP_TRUE,
		pwdutilsSLPRegReport,
		&callbackerr);

  /* err may contain an error code that occurred as the slp library    */
  /* _prepared_ to make the call.                                     */
  if ((err != SLP_OK) || (callbackerr != SLP_OK))
    {
      dbg_log ("Error registering service with slp %i", err);
      return -1;
    }

  /* callbackerr may contain an error code (that was assigned through */
  /* the callback cookie) that occurred as slp packets were sent on    */
  /* the wire */
  if( callbackerr != SLP_OK)
    {
      dbg_log ("Error registering service with slp %i",
	       callbackerr);
      return callbackerr;
    }

  /* Now that we're done using slp, close the slp handle */
  SLPClose (hslp);

  return 0;
}

int
deregister_slp ()
{
  SLPError err;
  SLPError callbackerr;
  SLPHandle hslp;

  if (url == NULL)
    {
      dbg_log ("URL not registerd!");
      return -1;
    }

  err = SLPOpen ("en", SLP_FALSE, &hslp);
  if(err != SLP_OK)
    {
      dbg_log ("Error opening slp handle %i", err);
      return -1;
    }

    /* DeRegister a service with SLP */
  err = SLPDereg (hslp, url, pwdutilsSLPRegReport, &callbackerr);

  free (url);
  url = NULL;

  /* err may contain an error code that occurred as the slp library    */
  /* _prepared_ to make the call.                                     */
  if ((err != SLP_OK) || (callbackerr != SLP_OK))
    {
      dbg_log ("Error unregistering service with slp %i", err);
      return -1;
    }

  /* callbackerr may contain an error code (that was assigned through */
  /* the callback cookie) that occurred as slp packets were sent on    */
  /* the wire */
  if( callbackerr != SLP_OK)
    {
      dbg_log ("Error registering service with slp %i",
	       callbackerr);
      return callbackerr;
    }

  /* Now that we're done using slp, close the slp handle */
  SLPClose (hslp);

  return 0;
}

#endif
